//--- ms_model.cpp ------------------------------------------------------------------------
// This file is used in the build of ms_model.dll, one of the two modules that form the
// application, Ms Resi.  It contains the core code of the physological model, developed
// from MacPuf. MacPuf is detailed in the monograph,
// "A Computer Model of Human Respiration", C. J.Dickinson, MTP Press, 1977.
// Further text on authorship is available at run time.
// Mal Haysom, Civil Engineering and Physical Sciences, La Trobe University, December 2009

// --- version history --------------------------------------------------------------------
// 19/05/2009 errors spotted by Michelle Gibson corrected
// 24/05/2009 transient on re-run corrected
// 01/06/2009 delay() look-up table structured
// 04/06/2009 'minor' changes from Version 17 12/1981 George Havenith included, noted as GH
// 11/07/2009 WRATE introduced PD/WRATE relationship as per GH
// 11/07/2009 GH catabolism and TRQ code included, delay() implemented with interpol()
// 17/08/2009 time display on clinical comments improved
// 02/09/2009 damp() macro replaced by time constant based damp() function (noted as TC)
// 02/09/2009 change at M1241 to improve stability at low oxygen levels
// 26/09/2009 death parameters system introduced
// 26/09/2009 error in initialization of C[71] corrected
// 01/11/2009 DLL system introduced, structural changes
// 04/12/2009 V1.00 posted

//--- includes ----------------------------------------------------------------------------
#include <math.h>
#include <string.h>
#include "ms_variable.h"             // variable structure definition and utility functions
#include "ms_subject.h"         // clinical codes and the wherewithal to make a new subject

//--- version defines ---------------------------------------------------------------------
#define MDL_VER	"1.00"				     // version number of this model module
#define	CMMNT	"standard version "__DATE__                 // enter developer comment here

// --- local bager() origins --------------------------------------------------------------
// (see ms_variable.h for external origins)
#define MM1	11					    // call from model() circa M781
#define MM2	12					    // call from model() circa M820
#define MM3	13					    		   // not allocated
#define MM4	14					    		   // not allocated
#define MM5	15					                   // not allocated
#define MM6	16					                   // not allocated

// --- local physiological functions ------------------------------------------------------
void gases(float, float, float, float *, float *, float *);       // determines gas content
void gsinv(float, float *, float *, float *, float *, float *);   // determine gas partials
void delay(float coadj); // uses delay history to present delayed values for some variables
bool constant(int NREPT);                  // computes all parameters constant during a run

// --- local utility functions ------------------------------------------------------------
float damp(float in, float crnt, float tc);          // one iteration in single pole filter
float interpol(float in, struct coord *pcd, int sz);   // interpolates y value from x value
int fold(int index);   // takes index to delay buffer and if necessary adjusts for wrapping

// --- interpolation tables ---------------------------------------------------------------
struct coord {float x, y;};  	         // coordinate structure for interpolation function

// delay table (cardiac output, circulation time)
struct coord dly_dt[] = {1.5,25.0, 2.5,22.5, 5.0,20.0, 11.5,16.0, 25.0,10.0};
#define sz_dly_dt (sizeof(dly_dt)/(2*sizeof(float)))      // delay table, number of entries
#define NUM_DLY 60			       		   // size of delay circular buffer

// workrate table GH (work load, efficiency)
struct coord wr_dt[] = {0.0,0.0, 3.0,2.0, 15.0,6.7, 30.0,11.5, 45.0,14.5, 75.0,18.0,
                                90.0,19.3, 120.0,20.7, 150.0,20.9, 225.0,22.5, 300.0,20.5};
#define sz_wr_dt (sizeof(wr_dt)/(2*sizeof(float)))        // WRATE table, number of entries

// tissue respiratory quotient table GH
struct coord trq_dt[] = {250.0,0.80, 450.0,0.86, 1320.0,0.87, 2100.0,0.90, 3150.0,1.00};
#define sz_trq_dt (sizeof(trq_dt)/(2*sizeof(float)))        // TRQ table, number of entries

// muscle lactate catabolism table GH
struct coord ctx_dt[] = {450.0,0.270, 578.0,0.280, 626.0,0.402, 730.0,1.990, 734.0,2.00,
			 860.0,8.000, 1010.0,14.0, 1130.0,23.5, 2500.0,5.50, 3500.0,0.35};
#define sz_ctx_dt (sizeof(ctx_dt)/(2*sizeof(float)))      // CATOX table, number of entries

// --- flags, indexes ---------------------------------------------------------------------
bool NARTI = true;                          // ventilation, true: natural, false: artifical
int PL = STD;                                                          // respiration index

// --- passport parameters ----------------------------------------------------------------
trade variable height = {"height", 0.0, 95.0, 200.0, "cm"};
#define HT height.vlu
trade variable weight = {"weight", 0.0, 20.0, 200.0, "kg"};
#define WT weight.vlu
trade variable age =    {"age",    0.0,  8.0, 100.0, "year"};
#define AGE age.vlu
trade int sex = 1;                                                  // 0 = female, 1 = male

// --- subject factors --------------------------------------------------------------------
trade variable factor[] = {
{"2,3-DPG concentration in red cells",               0.0,  1.0,  15, "mmol/litre"},
  #define DPG   factor[0].vlu	// (23)
{"Addition of bicarbonate or acid, standard bicarb.",0.0, -300, 300, "mmol"},
  #define ADDC3 factor[1].vlu	// (21)
{"Bag CO2 percentage",                               0.0,  0.0,  25, "%"},
  #define BAGCP factor[2].vlu	//  na
{"Bag O2 percentage",                                0.0,  0.0, 100, "%"},
  #define BAGOP factor[3].vlu	//  na
{"Bag volume",                                       0.0,  300, 30000, "ml BTPS"},
  #define BAGV  factor[4].vlu	// (116)
{"Body temperature",                                 0.0,   25,  44, "degree centigrade"},
  #define TEMP  factor[5].vlu	// (14)
{"Brain bicarbonate, deviation from normal (+/-)",   0.0,  -15,  20, "mmol/litre"},
  #define BC3AJ factor[6].vlu	// (22)
{"Breathing capacity",                               0.0,  0.0, 500, "% average normal"},
  #define PR    factor[7].vlu	// (24)
{"Cardiac pump performance",                         0.0,  0.0, 200, "% normal"},
  #define CO    factor[8].vlu	// (3)
{"Central neurogenic (learnt) respiratory drive",    0.0,  0.0, 200, "% normal"},
  #define CZ    factor[9].vlu	// (12)
{"Extra anatomical right-to-left shunt",             0.0,  0.0,  90, "% cardiac output"},
  #define FADM  factor[10].vlu	// (5)
{"Extra dead space (above normal value)",            0.0,  0.0, 100, "ml BTPS"},
  #define BULLA factor[11].vlu	// (6)
{"Haemoglobin blood density",                        0.0,  0.0,  25, "g/100 ml"},
  #define HB    factor[12].vlu	// (18)
{"Index of state of physical fitness",               0.0,   27,  43, ""},
  #define FITNS factor[13].vlu	// (25)
{"Inspiratory/total breath duration ratio",          0.0,  0.0, 1.0, ""},
  #define SPACE factor[14].vlu	// (26)
{"Inspired CO2 percentage",                          0.0,  0.0, 100.0, "%"},
  #define FIC2  factor[15].vlu	// (2)
{"Inspired O2 percentage",                           0.0,  0.0, 100.0, "%"},
  #define FIO2  factor[16].vlu	// (1)
{"Left-to-right shunt, ratio to cardiac output",     0.0,  0.0, 5.0, ""},
  #define SHUNT factor[17].vlu	//  (28)
{"Lung elastance",                                   0.0,  0.0,  90, "cm water/litre"},
  #define ELAST factor[18].vlu	// (8)
{"Lung volume, (end-expiration)",                    0.0,  0.0, 15000, "ml BTPS"},
  #define VLUNG factor[19].vlu	// (7)
{"Maximum cardiac output",                           0.0,  0.0,  50, "litre/min"},
  #define COMAX factor[20].vlu	// (27)
{"Packed cell volume",                               0.0,  0.0,  80, "%"},
  #define PCV   factor[21].vlu	// (19)
{"Positive end-expiratory pressure",                 0.0,  0.0,  50, "cm water"},
  #define PEEP  factor[22].vlu	// (30)
{"Tissue ECF distribution volume",                   0.0,  5.0,  50, "litres"},
  #define TVOL  factor[23].vlu	// (17)
{"Tissue respiratory quotient",                      0.0, 0.0, 15, "CO2 output/O2 uptake"},
  #define TRQ   factor[24].vlu	// (15)
{"Total barometric pressure,",                       0.0,   50, 3800, "mm Hg"},
  #define BARPR factor[25].vlu	// (13)
  #define AF_0 25
{"Venous admixture effect",                          0.0,  0.0,  90, "% cardiac output"},
  #define VADM  factor[26].vlu	// (9)
{"Venous blood volume",                              0.0,  500, 10000, "ml"},
  #define VBLVL factor[27].vlu	// (20)
{"Ventilatory response to CO2 or H+",                0.0,  0.0, 500, "% average normal"},
  #define AZ    factor[28].vlu	// (10)
{"Ventilatory response to failing partial O2",       0.0,  0.0, 500, "% average normal"},
  #define BZ    factor[29].vlu	// (11)
{"Vital capacity",                                   0.0,  0.0, 10.0, "litres BTPS"},
  #define VC    factor[30].vlu	// (29)
{"Work Rate",					     0.0,  0.0, 200, "Watt"}
  #define WRATE	factor[31].vlu	// (4)
                    };

// --- physiological variables ------------------------------------------------------------
trade variable display[] = {
{"Alveolar CO2 amount",                              0.0, 0.0,  200, "ml STPD"},
  #define AC2MT display[0].vlu	// (40)
{"Alveolar CO2 partial pressure",                    0.0, 0.0,   75, "mm Hg"},
  #define AC2PR display[1].vlu	// (42)
  #define AD_0 1
{"Alveolar N2 amount",                               0.0, 0.0,  500, "ml STPD"},
  #define AN2MT display[2].vlu	// (65)
{"Alveolar O2 amount",                               0.0, 0.0,  500, "ml STPD"},
  #define AO2MT display[3].vlu	// (39)
{"Alveolar O2 partial pressure",                     0.0, 0.0,  150, "mm Hg"},
  #define AO2PR display[4].vlu	// (41)
  #define AD_1 4
{"Alveolar ventilation",                             0.0, 0.0, 20.0, "litres/mm BTPS"},
  #define FVENT display[5].vlu	// (35)
{"Arbitrary index of risk of decompression symptoms",0.0, 0.0,  200, ""},
  #define BUBBL display[6].vlu	// (83)
{"Arterial bicarbonate content",                     0.0, 0.0,   50, "mmol/litre"},
  #define RC3CT display[7].vlu	// (60)
{"Arterial CO2 amount",                              0.0, 0.0, 1000, "ml STPD"},
  #define RC2MT display[8].vlu	// (63)
{"Arterial CO2 content",                             0.0, 0.0,  100, "ml/100 ml"},
  #define RC2CT display[9].vlu	// (78)
{"Arterial CO2 partial pressure",                    0.0, 0.0,   75, "mm Hg"},
  #define RC2PR display[10].vlu	// (74)
  #define AD_2 10
{"Arterial lactate concentration",                   0.0, 0.0, 10.0, "mmol/litre"},
  #define RLACT display[11].vlu	// (90)
{"Arterial N2 partial pressure",                     0.0, 0.0,  150, "mm Hg"},
  #define RN2MT display[12].vlu	// (108)
  #define AD_3 12
{"Arterial O2 amount",                               0.0, 0.0, 1000, "ml STPD"},
  #define RO2MT display[13].vlu	// (62)
{"Arterial O2 content",                              0.0, 0.0,  150, "ml/100 ml"},
  #define RO2CT display[14].vlu	// (49)
{"Arterial O2 partial pressure",                     0.0, 0.0,  150, "mm Hg"},
  #define RO2PR display[15].vlu	// (72)
  #define AD_4 15
{"Arterial O2 saturation (maximum)",                 0.0, 0.0,  100, "%"},
  #define PJ    display[16].vlu	// (76)
{"Arterial pH",                                      0.0, 7.0,  8.0, ""},
  #define RPH   display[17].vlu	// (33)
{"Arterial pool CO2 content of blood leaving",       0.0, 0.0,  100, "ml/100 ml"},
  #define EC2CT display[18].vlu	// (101)
{"Arterial pool N2 content of blood leaving",        0.0, 0.0, 10.0, "ml/100 ml"},
  #define EN2CT display[19].vlu	// (106)
{"Arterial pool O2 content of blood leaving",        0.0, 0.0,  100, "ml/100 ml"},
  #define EO2CT display[20].vlu	// (94)
{"Bag CO2 amount",                                   0.0, 0.0, 1000, "ml STPD"},
  #define BAGC  display[21].vlu	// (38)
{"Bag O2 amount",                                    0.0, 0.0, 1000, "ml STPD"},
  #define BAGO  display[22].vlu	// (37)
{"Brain blood flow",                                 0.0, 0.0,  200, "ml/100 g brain/min"},
  #define CBF   display[23].vlu	// (68)
{"Brain CO2 amount",                                 0.0, 0.0, 1000, "ml STPD"},
  #define BC2MT display[24].vlu	// (67)
{"Brain CO2 content of blood leaving",               0.0, 0.0,  100, "ml/100 ml"},
  #define BC2CT display[25].vlu	// (58)
{"Brain CO2 partial pressure of blood leaving",      0.0, 0.0,  100, "mm Hg"},
  #define BC2PR display[26].vlu	// (46)
  #define AD_5 26
{"Brain O2 amount",                                  0.0, 0.0,  100, "ml STPD"},
  #define BO2MT display[27].vlu	// (66)
{"Brain O2 content of blood leaving",                0.0, 0.0,  100, "ml/100 ml"},
  #define BO2CT display[28].vlu	// (57)
{"Brain O2 partial pressure of blood leaving",       0.0, 0.0,  150, "mm Hg"},
  #define BO2PR display[29].vlu	// (45)
  #define AD_6 29
{"Brain bicarbonate content",                        0.0, 0.0,  100, "mmol/litre"},
  #define BC3CT display[30].vlu	// (91)
{"Brain pH (at putative central chemoreceptor site)",0.0, 7.0,  8.0, ""},
  #define BPH   display[31].vlu	// (36)
{"Cardiac output (actual effective)",                0.0, 0.0, 20.0, "litre/mm"},
  #define COADJ display[32].vlu	// (93)
{"Dead space (total effective physiological)",       0.0, 0.0,  500, "ml BTPS"},
  #define DSPAC display[33].vlu	// (70)
{"Excess N2 held above normal maximum saturation",   0.0, 0.0, 10.0, "ml STPD"},
  #define UN2MT display[34].vlu	// (107)
{"Heart rate",					     0.0, 20,   220, "BPM"},
  #define HRATE	display[35].vlu	// (109)
{"Metabolic rate",                                   0.0, 0.0,  500, "% resting value"},
  #define PD    display[36].vlu	// (81)
{"Pulmonary capillary (idealized) blood CO2 content",0.0, 0.0,   75, "ml/100 ml"},
  #define PC2CT display[37].vlu	// (53)
{"Pulmonary capillary (idealized) blood O2 content", 0.0, 0.0,  150, "ml/100 ml"},
  #define PO2CT display[38].vlu	// (54)
{"Respiratory quotient",                             0.0, 0.0,  5.0, ""},
  #define RQ    display[39].vlu	// (69)
{"Respiratory rate",                                 0.0, 0.0,   50, "cycles/min"},
  #define RRATE display[40].vlu	// (48)
{"Tidal volume",                                     0.0, 0.0,  2500, "ml BTPS"},
  #define TIDVL display[41].vlu	// (47)
{"Tissue bicarbonate content of blood leaving",      0.0, 0.0,   75, "mmol/litre"},
  #define TC3CT display[42].vlu	// (87)
{"Tissue CO2 content of blood leaving",              0.0, 0.0,   75, "ml/100 ml"},
  #define TC2CT display[43].vlu	// (56)
{"Tissue CO2 partial pressure",                      0.0, 0.0,   75, "mm Hg"},
  #define TC2PR display[44].vlu	// (97)
  #define AD_10 44
{"Tissue CO2 stores",                                0.0,  5.0,  50, "litre STPD"},
  #define TC2MT display[45].vlu	// (16)
{"Tissue lactate amount (body total)",               0.0, 0.0,   50, "mmol"},
  #define TLAMT display[46].vlu	// (89)
{"Tissue N2 amount in fast compartment",           0.0, 0.0,  100, "ml STPD"},
  #define TN2MT display[47].vlu	// (102)
{"Tissue N2 amount in slow compartment",           0.0, 0.0, 4000, "ml STPD"},
  #define SN2MT display[48].vlu	// (112)
{"Tissue N2 partial in fast compartment",          0.0, 0.0, 1000, "mm Hg"},
  #define TN2PR display[49].vlu	// (103)
  #define AD_8 49
{"Tissue N2 partial in slow compartment",          0.0, 0.0, 4000, "mm Hg"},
  #define SN2PR display[50].vlu	// (105)
  #define AD_9 50
{"Tissue O2 amount",                                 0.0, 0.0, 1000, "ml STPD"},
  #define TO2MT display[51].vlu	// (95)
{"Tissue O2 content of blood leaving",               0.0, 0.0,  200, "ml/100 ml"},
  #define TO2CT display[52].vlu	// (55)
{"Tissue O2 partial pressure (time shifted values)", 0.0, 0.0,  150, "mm Hg"},
  #define TO2PR display[53].vlu	// (96)
  #define AD_11 53
{"Tissue pH ",                                       0.0, 7.0,  8.0, ""},
  #define TPH   display[54].vlu	//(59)
{"Total ventilation",                                0.0, 0.0,   50, "litres/min BTPS"},
  #define DVENT display[55].vlu	// (51)
{"Venous admixture (total effective)",               0.0, 0.0,  100, "% cardiac output"},
  #define PW    display[56].vlu	// (80)
{"Venous bicarbonate amount",                        0.0, 0.0,   50, "mmol"},
  #define VC3MT display[57].vlu	// (86)
{"Venous bicarbonate content (mixed venous)",        0.0, 0.0,   50, "mmol/litre"},
  #define VC3CT display[58].vlu	// (88)
{"Venous CO2 amount",                                0.0, 0.0, 2000, "ml STPD"},
  #define VC2MT display[59].vlu	// (50)
{"Venous CO2 content (blood in pulmonary artery)",   0.0, 0.0,  100, "ml/100 ml"},
  #define VC2CT display[60].vlu	// (61)
{"Venous O2 amount",                                 0.0, 0.0, 2000, "ml STPD"},
  #define VO2MT display[61].vlu	// (98)
{"Venous O2 content of blood in pulmonary artery",   0.0, 0.0,  100, "ml/100 ml"},
  #define VO2CT display[62].vlu	// (31)
{"Venous pH (of mixed venous blood)",                0.0, 7.0,  8.0, ""},
  #define VPH   display[63].vlu	// (34)
{"No display",                                         0,   0,   0, ""}
                    };
#define NUM_FCT	sizeof(factor)/sizeof(factor[0])               // number of subject factors
#define NUM_DSY sizeof(display)/sizeof(display[0])           // number of display variables

trade variable alt_fact[] = {    // the variables in factor[] with alternate pressure units
{"Total barometric pressure",                          0.0, 6.0, 510, "kPa"}    // 0   (13)
                      };
trade int a_f_lst[] = {AF_0, -1};                  // array of factor indexes for the above

trade variable alt_dspy[] = {   // the variables in display[] with alternate pressure units
{"Alveolar CO2 partial pressure",                      0.0, 0.0,  10, "kPa"},   //  0  (42)
{"Alveolar O2 partial pressure",                       0.0, 0.0,  20, "kPa"},   //  1  (41)
{"Arterial CO2 partial pressure",                      0.0, 0.0,  10, "kPa"},   //  2  (74)
{"Arterial N2 partial pressure",                       0.0, 0.0,  20, "kPa"},   //  3 (108)
{"Arterial O2 partial pressure",                       0.0, 0.0,  20, "kPa"},   //  4  (72)
{"Brain CO2 partial pressure of blood leaving",        0.0, 0.0,  15, "kPa"},   //  5  (46)
                                                                                //  7
{"Brain O2 partial pressure of blood leaving",         0.0, 0.0,  20, "kPa"},   //  6  (45)
{"Tissue N2 partial in fast compartment",            0.0, 0.0, 500, "kPa"},   //  8 (103)
{"Tissue N2 partial in slow compartment",            0.0, 0.0, 150, "kPa"},   //  9 (105)
{"Tissue CO2 partial pressure",                        0.0, 0.0,  10, "kPa"},   // 10  (97)
{"Tissue partial O2 (time shifted values)",  	       0.0, 0.0,  20, "kPa"}    // 11  (96)
                      };
trade int a_d_lst[] = {AD_0,AD_1,AD_2,AD_3,AD_4,AD_5,AD_6,AD_8,AD_9,AD_10,AD_11, -1};

// --- other ------------------------------------------------------------------------------
trade float other[NUM_OTH] = {0.0};        // variables, neither subject factor nor display 
#define   AVENT         other[0]                               // alveolar ventilation (99)
#define   BO2AD         other[1]                // index of brain oxygenation adequacy (92)
#define   CONOM         other[2]                     // nominal resting cardiac output (82)
#define   CONSO         other[3]       // tissue O2 consumption, nominal resting value (73)
#define   DPH           other[4]    // arterial H+ activity (at end of each iteration) (43)
#define   FEV           other[5]                             // forced expired volume (104)
#define   PG            other[6]    // index of time brain has been deprived of oxygen (75)
#define   QA            other[7]                      // nett O2 uptake per iteration (113)
#define   QB            other[8]                      // nett CO2 uptake per iteration (79)
#define   REFLV         other[9]   	                  // reference volume of lungs (71)
#define   RVADM         other[10]    // extra shunt effect for emphysematous subjects (114)
#define   SVENT         other[11]                 // total effective ventilatory drive (52)
#define   TC2RF         other[12]    // reference value detecting CO2 partial pressure (84)
#define   TC3AJ         other[13]                      // tissue bicarbonate adjusted (111)
#define   TC3MT         other[14]                         // tissue bicarbonate amount (85)
#define   XDSPA         other[15]      // extra dead space for altered function tests (115)
#define   XLACT         other[16]          		// local lactate concentration (44)
#define   XRESP         other[17]                // damping function respiratory drive (64)
#define   STRVL   	other[18]       		    // heart stroke volume GH (110)
#define   RCOAJ         other[19] 				    // arterial ... GH (32)

// --- macros -----------------------------------------------------------------------------
#define	phfnc(X,Y)      (6.10+log((X)/(0.03*(Y)))*0.434294482)

// --- local variables --------------------------------------------------------------------
float TDLAY[NUM_DLY][4];      // delay buffer, maxiumum delay NUM_DLY increments of 4 items

// --- variables for export ---------------------------------------------------------------
trade float history[NUM_DSY+NUM_OTH][HST_LNG] = {0}; // display and other variables history
trade char app_buf[];					         // clinical message buffer
trade char *comment = CMMNT;                 				 // version comment
trade char *mdl_ver = MDL_VER;                              // version number of this model

// --- export variables for respiration ---------------------------------------------------
trade bool *ventilation = &NARTI;
trade float *res_rate = &RRATE;
trade float *tidal = &TIDVL;
trade float *positive = &PEEP;
trade int *bag_index = &PL;
trade float *bag_vol = &BAGV;
trade float *bag_co2 = &BAGC;
trade float *bag_o2 = &BAGO;
trade int num_fct = NUM_FCT;                                  // number of factor variables
trade int num_dsy = NUM_DSY;                                 // number of display variables
trade int num_oth = NUM_OTH;                                   // number of other variables

// --- local variables --------------------------------------------------------------------
char app_buf[200] = "";						 // clinical message buffer
int crrnt_state;               	         // current structured set of clinical message bits
int prev_state;               	        // previous structured set of clinical message bits
int prv_mnr;               		 // previous occurrence of minor symptoms indicator
int prv_svr;              	        // previous occurrence of severe symptoms indicator
float C[76];              	      // an array of constants determined for a new subject

// --- model() ----------------------------------------------------------------------------
/* model() is the core of physiological code. It returns non-zero if there is a subject
problem, zero otherwise. */
int model(void)
	{
     	float FD, FTCO, FTCOC, FY, PC, PC2, PO2, SO2, U, V, W, X, XVENT, Y, Z;
        float CATCID, CATLIV, ORGCAT, CATMUS;    // catabolism kidney, liver, organ, muscle
        float CATOX;    	          // oxygen comsumpton/musle catbolism relationship
	int item = 0;						// non-zero indicates error
// P 191.4
        Y=RO2CT*0.056; 		           // increase cardiac output if O2 supplied is low
        if(Y >= 0.40)       // reduces cardiac output increase with hypoxia 0.35 -> 0.40 GH
            goto M240;
        Y=0.40;                                          	         // 0.35 -> 0.40 GH
/* COADJ is adjusted cardiac output, used mainly as index FTCO which takes account of
cardiac output per unit time. Output goes up as total O2 consumption increases. */
M240:   COADJ = damp((C[8]+C[9])/Y,COADJ,C[61]); 				      // GH
        X=-0.3*COADJ*COADJ+11.0*COADJ+30.0;    // calculate heart rate and stroke volume GH
        if(COADJ > 20.0)
            X=130.0;
        STRVL=C[75]*X/130000.0;				         // stroke volume in litres
        HRATE=COADJ/STRVL;
        if(HRATE < C[74])      // limit cardiac output through stroke volume and heart rate
            goto M242;
        HRATE=C[74];
        COADJ=HRATE*STRVL;
M242:   if(CO >= 3.0)
            goto M260;
// if heart stopped make output and heartrate zero to stop arterial composition changing
        COADJ=0.0;
        HRATE=0.0;
        goto M280;
M260:	if(COADJ-COMAX <= 0)    			    // limit maximum cardiac output
            goto M280;
        COADJ=COMAX;
// FTCO is number of 100 ml portions of blood circulating per fractional time interval
M280: 	FTCO=C[17]*COADJ;
        FTCOC=FTCO*(1.0-C[69]*COADJ);
// O2 contentration or pressure influences venous admixture and (very slowly) 2,3-DPG
        DPG=DPG+(C[71]-RO2CT-DPG)*C[72];       // new parameters amending rate of change GH 
        X=AO2PR;
        if(X <= 200.0)
            goto M300;
        X=200.0;
M300: 	Y=AO2PR;  	        // increase effective venous admixture if alv.PO2 very high 
        if(Y <= 600.0)
            goto M320;
        Y=600.0;
M320: 	if(AO2PR <= 400.0)
            goto M340;
        X=X-(Y-400.0)*0.3;
M340:	if(X > 55.0)
           goto M360;
        X=55.0;
/* PW = effective venous admixture, affected by PEEP, alveolar PO2 etc and also 
incorporating a fixed shunt component, FADM  */
M360: 	PW=(C[18]/X+C[19])*C[21]+FADM;
        if(PW <= 100.0) 		                 // limit admixtures exceeding 100%
             goto M380;
        PW=100.0;
M380: 	X=PW*0.01;
// P 192 
        PC=1.0-X;
/* arterial CO2 + O2 amounts incremented by mixture of pure venous and pure idealized
pulmonary cap. blood determined by ratios X and PC nitrogen content is determined in terms
of partial pressures */
        RN2MT=RN2MT+FTCO*((X*TN2PR+PC*(C[11]-AO2PR-AC2PR))*0.00127-EN2CT);
        U=X*VC2CT+PC*PC2CT;
        V=X*VO2CT+PC*PO2CT;
        RC2MT=RC2MT+FTCO*(U-EC2CT);
        RO2MT=RO2MT+FTCO*(V-EO2CT);
        W=C[22]*COADJ; 	     // contents passing to tissues affected by rates of blood flow 
        if(COADJ<0.25)    // if heart stopped prevent changes in arterial blood composition
            goto M410;
	EO2CT=damp(RO2MT*0.1,EO2CT,W);
        EC2CT=damp(RC2MT*0.1,EC2CT,W);
        EN2CT=damp(RN2MT*0.1,EN2CT,W);
M410:   Z=1/(COADJ*C[17]);
	RO2CT=damp(V,RO2CT,Z); 	       // RC2CT is content of blood reaching chemoreceptors 
        RC2CT=damp(U,RC2CT,Z);
/* O2CON and C2CON are used every time for entering gases(). Before entering gases() set
contents of O2 + CO2 for arterial blood.
Same for bicarb. (H2CO3), which has to take account of in vitro influence of arterial PCO2
on bicarb. concentration so that pH can be calculated. */
        RC3CT=C[3]*(RC2PR-40.0)+VC3MT*C[1];
        if(RC3CT <= 0)
            {
            item = 1;
            goto M_END;
            }
       RPH=phfnc(RC3CT,RC2PR);   // enter art.bicarb. calculate pH and enter value into RPH
       gsinv(RPH,&RO2PR,&RC2PR,&RO2CT,&RC2CT,&SO2);
       PJ=SO2*100.0;  			         // store arterial saturation as percentage
/* U is energy expenditure from 'metabolic rate' specified by operator. 1st=O2 consumption
of resp. msmuscles. 2nd=O2 cons. of heart. 3rd=rest of body. In the event of anaerobic
metab., same energy requirements involve 11X number of moles of lactate produced with xlact
O2 spared */
	U=FT*(pow(fabs(SVENT),C[4])*C[5]+COADJ+C[7]);
   	TO2MT=TO2MT+FTCOC*(EO2CT-TO2CT)-U+XLACT;  // compute new tissue gas amounts (T-2MT)
        if(TO2MT <= 0)
            {
            item = 2;
	    goto M_END;
            }
        TO2PR=damp(TO2MT*C[31],TO2PR,C[55]);    // compute tissue PO2 damping appropriately
        X=TO2MT-250;
        if(X <= 0)
            goto M530;
// next section concerns lactic acid metabolism
        TO2PR=45.0+0.09*X;
// local variable Y used later for catabolism related to cardiac output and metabolism
M530:   Y=RLACT*C[29];
/* Z=catabolic rate for lactate. X is threshold when TPH less than 7.0 catabolism impaired
cerebral blood flow (CBF) is used as described below (empirically) */
        W=CBF*0.019;
        if(W <= 1.0)
            goto M536;
        W=1.0;
M536: 	X=TPH*10.0-69.0;
        if(X <= W)
           goto M550;
        X=W;
/* 1st term is hepatic removal. 2nd is renal removal with pH influence 3rd term is blood
flow related metabolism by muscles, made dependent on cardiac output (COADJ). Whole
expression is multiplied by Y, a function of blood lactate concentration. */
/* lactate catabolism split into organs and muscle catabolism (catliver and catcidney)
and catmus GH */
M550: 	CATLIV=Y*(X*0.8612);           			          // refer GH April page 52
	CATCID=Y*(0.0232*pow(2,(8.0-TPH)*3.33));
	ORGCAT=CATCID+CATLIV;
	X=ORGCAT-0.3;
	if(X > 0.0)
            ORGCAT=0.30;
	X=QA/FT;
      	CATOX=interpol(X, ctx_dt, sz_ctx_dt);
	CATMUS=Y*CATOX;                // catox is a catabolism-oxygen consumption relation
	Z=ORGCAT+CATMUS;
/* local variable W above makes slight allowance for reduced liver blood flow; which causes
decreased metabolism and increased production when there is a low PCO2 or alkalosis.
Cerebral blood flow (CBF) is used for computation since it is computed elsewhere and
changes in the appropriate way. */
// P 193 
      	W=C[70]/(W+0.3);
// FITNS is threshold for switch to anaerobic metabolism related to fitness
        V=C[73]-TO2PR;    					       // FITNS -> C[73] GH
        if(V <= 0)
            goto M570;
// provide virtual trigger effect, rapidly increase lactic acid prod. if tissue PO2 low
	W=W+C[42]*pow(V+1.0,4.0);
        Z=Z*TO2PR*0.04;           	      // catabolismi(Z) falls if tissue PO2 too low
/* XLACT is O2 sparing effect of lactic acid production (not allowed to exceed actual O2
consumption) */
M570: 	X=2.04*(W-C[32]);
        if(X <= U)
           goto M590;
        X=U;
M590: 	XLACT=damp(X,XLACT,C[53]);
/* limit of rate of lactate form determined by metab. drive to tissues (i.e. level of
exercise), and to body size */
	if(W <= C[40])
            goto M610;
        W=C[40];
/* reduce lactate catabolism if card. output less than 1/3 normal to take account of
probable dim. liver and kidney blood flows */
M610:   X=C[24]-COADJ;
        if(X <= 0)
            goto M630;
	Z=Z*COADJ/C[24];
M630: 	V=W-Z; 	  // increment total lactic acid by difference between prod. and catabolism
        TLAMT=TLAMT+V;
        RLACT=damp(TLAMT*C[15],RLACT,C[55]/COADJ);
/* next is for nitrogen stores in tissues move N2 between fast (T) and slow (S) tissue
compartments according to partial pressure differences. */
	X=(TN2PR-SN2PR)*C[60];
        TN2MT=TN2MT+FTCOC*(EN2CT-TN2PR*0.00127)-X;
        SN2MT=SN2MT+X;
        Y=(SN2MT*C[26]-C[11])*C[27];     	       // test if slow space supersaturated
        if(Y > 0)
            goto M660;
// if so, augment U and decrement S, or vice versa if ambient pressure relatively higher
        Y*=0.3;
        if(UN2MT <= 0)
            goto M670;
M660:   SN2MT=SN2MT-Y;
/* BUBBL is arbit.index of bubble sympts, taking into account BTPS volume and loading by
number of molecules of gas. BUBBL is only a rough index - under development  */
        BUBBL=pow(UN2MT,1.2)*C[23];
        UN2MT=UN2MT+Y;
M670: 	SN2PR=SN2MT*C[26];     				       // compute partial pressures
        TN2PR=TN2MT*C[28];
      	X=QA/FT;
     	TRQ = interpol(X, trq_dt, sz_trq_dt);    // TRQ and oxygen consumption relationship 
// tissue CO2 exchanges; U=metab.; 0.001 converts cc to litres */
        TC2MT=TC2MT+(FTCOC*(EC2CT-TC2CT)+TRQ*U)*0.001;
// compute partial pressures from total CO2 and stand. bicarb.
       	TC2PR=(TC2MT*C[30]-TC3MT*C[36]+C[33])*C[43];
// FY stores change in tissue CO2 for adjusting tissue HCO3 buffers
        FY=TC2PR-TC2RF;
        TC2RF=TC2PR;
/* 1.0 (GH, increase of bicarbonate buffering effect) in line below represents buffers of
lactic acid partly inside cells. so that the displacement of bicarbonate is less than
strict molar equivalence */
        TC3MT=TC3MT+FTCOC*0.1*(VC3MT*C[1]-TC3MT*C[13])-1.0*V;
        Y=(TC2PR-40.0)*C[3];
	TC3CT=TC3MT*C[13]+Y;
        if(TC3CT <= 0)
            {
            item = 3;
            goto M_END;
            }
	TPH=phfnc(TC3CT,TC2PR);
// P 194 
       gases(TPH,TO2PR,TC2PR,&TO2CT,&TC2CT,&SO2);
/* amounts of gases in venous pool incremented by arriving blood from tissues and
decremented by bloodgoing to lungs, same for bicarbonate, contents V-2CT then determined */
        X=C[69]*COADJ*FTCO;
        VC2MT=VC2MT+FTCOC*TC2CT-FTCO*(VC2CT*C[14]-RC2CT*SHUNT)+X*EC2CT;
        VO2MT=VO2MT+FTCOC*TO2CT-FTCO*(VO2CT*C[14]-RO2CT*SHUNT)+X*EO2CT;
        VC3MT=VC3MT+FTCOC*0.1*(TC3MT*C[13]-VC3MT*C[1])+ADDC3;
        X=TC3AJ*C[10];
        TC3AJ=TC3AJ-ADDC3+V-X;
        TC3MT=TC3MT-FY*C[64]+X*0.67;
        VO2CT=VO2MT*C[2];
        VC2CT=VC2MT*C[2];
// delay can be omitted with loss of accuracy only when using short iteration intervals
        delay(COADJ);
        VC3CT=VC3MT*C[1]+Y;
        if(VC3CT <= 0)
            {
            item = 4;
            goto M_END;
            }
        VPH=phfnc(VC3CT,TC2PR);
/* next long section concerns gas exch.in lungs. PC is fraction of cardiac output perfectly
mixed with alveolar gases. U and V = amounts of each gas taken in per iteration period */
 	PC=FTCO*C[14]*PC;
        X=AVENT*C[12];
        U=X*FIO2;
        V=X*FIC2;
/* next 3 sts. compute new amounts of each gas in lungs. W is volume at end of nominal
'inspiration'  */
        AO2MT=AO2MT+U;
        AC2MT=AC2MT+V;
        Z=100.0-FIO2-FIC2;
        AN2MT=AN2MT+X*Z;          				      // rationalization GH
        W=AO2MT+AC2MT+AN2MT;
// now calculate alveolear partial pressures
        X=C[11]/W;
        PO2=AO2MT*X;
        PC2=AC2MT*X;
/* change alveolar gas amounts in accordance with blood gas contents entering(V-2CT) and
leaving(P-2CT) the lungs. PC=final new amount of total gas at end of all. */
        AO2MT=AO2MT+PC*(VO2CT-PO2CT);
        AC2MT=AC2MT+PC*(VC2CT-PC2CT);
        AN2MT=AN2MT+PC*(TN2PR*0.00127-EN2CT);
        PC=AO2MT+AC2MT+AN2MT;
/* FY becomes +ve only if more gas goes out than in, in which case FY is later brought into
the calculation of effective dead space */
        if(PL == CXA)
             goto M770;
        if(AVENT < 20.0)
            goto M770;
        FY=(PC-W)*C[34]/RRATE;
        goto M780;
M770: 	FY=0;
M780:	if(PL == GTC)          						  // glottis closed
 	    bager(MM1, &PC, &X);
// XVENT is volume exhaled in iteration period down to resting lung volume
	XVENT=PC*C[35]-VLUNG;
        if(XVENT < 0)             // if XVENT is -ve (rare) treat for diffusion respiration 
            {
            FD=XVENT*C[12];
            AO2MT=AO2MT-FD*FIO2;
            AC2MT=AC2MT-FD*FIC2;
            AN2MT=AN2MT-FD*Z;
            goto M850;
            }
	X=XVENT*C[25]; 								      // GH 
// P 195
	DVENT=X/PC;    	        						      // GH
        X=X/C[11];                                                                    // GH 
        Y=X*AO2PR;                                                                    // GH
        Z=X*AC2PR;                                                                    // GH 
        PC=X*(C[11]-AO2PR-AC2PR);                                                     // GH
// U=O2, V=CO2 uptake in iteration time. Y,Z,PC=gas outputs
// algebraic summing of intake and output of O2 and CO2
        QA=U-Y;
        QB=Z-V;
        if(PL == STD || PL == GTC)
            goto M830;
        bager(MM2, &XVENT, &DVENT);
        if(DVENT+9000.0 < 0)
            {
            item = 5;
            goto M_END;
            }
M830:   if(AVENT <= 0.0)              			 // prevent division by zero errors
           {
           QA=0.001;
           QB=0;
           }
	AO2MT=AO2MT-Y;      // set new amounts of each gas then calculate partial pressures 
        AC2MT=AC2MT-Z;
        AN2MT=AN2MT-PC;
M850:	U=C[11]/(AO2MT+AN2MT+AC2MT);
        V=C[37]/RRATE;                           // take account of insp/exp duration ratio 
        if(V <= 4.0)
            goto M870;
 	V=4.0;
M870:	if(AVENT >= 20.0)
            goto M890;
        V=0.0;
  // rate of change of alveolar gas tensions, function of tidal volume
M890:	X=C[38]/(TIDVL+100.0);
        Y=AO2MT*U;                            // compute end-'expiratory' partial pressures
        Z=AC2MT*U;
// damp function used to prevent oscillatory swings of alveolar gas pressures
        AO2PR=damp((Y+(PO2-Y)*V),AO2PR,X);
        AC2PR=damp((Z+(PC2-Z)*V),AC2PR,X);
        if(AO2PR <=0)
            {
            item = 6;
            goto M_END;
            }
	if(AC2PR <= 0)
            {
            item = 7;
            goto M_END;
            }
/* determine expired RQ then alveolar gas tensions and eventually contents of CO2 + O2
in pulmonary capillary blood (P-2CT) */
	if(QA == 0.0)
            goto M930;
        RQ=QB/QA;                                                   // respiratory quotient
M930:	X=VC3MT*C[1]+C[3]*(AC2PR-40.0);
        if(X < 0)
            {
            item = 8;
            goto M_END;
            }
	Y=phfnc(X,AC2PR);
        gases(Y,AO2PR,AC2PR,&PO2CT,&PC2CT,&SO2);
/* determine cerebral blood flow adjustments in relation to cardiac output and brain pH
(PCO2 sensitive) */
        Z=sqrt(COADJ)*0.5;
        if(Z <= 1.0)
            goto M970;
        Z=1.0;
M970:	Y=(7.4-BPH)*(BC2PR*0.0184-BC3AJ*0.1);
       	if(Y <= 0)
           goto M990;
        Y=300.0*Y*Y;
M990: 	if(Y <= 4.4)
            goto M1010;
        Y=4.4;
M1010:  CBF=damp((Y-0.12)*42.5*Z,CBF*Z,C[55]);
/* comp. brain gas amounts by metab.assuming RQ of 0.98 and allowing for different amounts
supplied in arterial blood and leaving in venous blood. Check for arithmetical errors then
calculate brain gas tensions from guestimated disassocation curves. */
        Y=CBF*C[39];
        X=C[41]*(BO2AD+0.25);
        Z=X;
        if(BO2PR > 18.0)
            goto M1040;
	Z=X*(BO2PR*0.11-1.0);
        X=X*(19.0-BO2PR);
        if(Z >= 0)
            goto M1040;
        Z=0.0;
M1040:  BO2MT=BO2MT+Y*(RO2CT-BO2CT)-2.0*Z*(BO2AD+0.1);
        if(BO2MT > 0)
           goto M1060;
        BO2MT=0.1;
// P 196
M1060:	BC2MT=BC2MT+Y*(RC2CT-BC2CT)+2.15*X;
        BO2PR=BO2MT*1.6;
        BC2PR=BC2MT*0.078;
        W=BC2PR-40.0;
        Y=BC3CT+BC3AJ+0.2*W;
/* a small proportion of added bicarbonate is added also to CFS, thus affecting breathing
appropriately */
        BC3AJ=BC3AJ+((RC3CT-24.0)*0.3-BC3AJ)*C[42];
/* adjust bicarb.to PCO2 then calculate 'brain pH' ie,pH at receptor, then proceed to 
determine contents of O2 and CO2 in blood leaving brain */
        X=phfnc(2.0*Y+RC3CT,2.0*BC2PR+RC2PR);
        Z=(pow(fabs(X-BPH)*100.0,2.0)+0.04);
        if(Z <= C[17])
            goto M1080;
        Z=C[17];
M1080:	BPH=damp(X,BPH,1/Z); 		    // restrict rate of change of brain receptor pH
	Z=phfnc(VC3MT*C[1]+(BC2PR-40.0)*C[3],BC2PR);
        gases(Z,BO2PR,BC2PR,&BO2CT,&BC2CT,&SO2);
/* now follow ventilation calculations, starting with NARTI=0 which is artificial 
ventilation compute total dead space:, anat + physiol., then alv. ventilation (AVENT) */
        if(NARTI != 0)
            goto M1100;
        DVENT=C[51];
/* nat.vent.controls. total vent (U and SVENT)=sum of central CO2 (PH) chemoreceptor drive
{slope X, intercept Z) O2 lack receptor (slope Y) central neurogenic drive (prop. to O2
consumption C[7], + constant etc).  AZ etc. are for manual adjustments of ventilatory
controls. */
M1100:	Y=(118.0-PJ)*0.05;
        Z=Y*0.002;
        X=(C[65]+Z-BPH)*1000.0*Y;
        if(X >= 0)
            goto M1120;
        X=0.0;
M1120:	W=(C[66]+Z-BPH)*150.0*Y;
/* high brain pH or low PCO2 only inhibits ventilation if learnt intrinsic drive (CZ) is
reduced or absent */
        if(W >= 0)
            goto M1150;
        if(C[67] <= 0)
            goto M1150;
        W=0.0;
M1150:	Z=(BC2PR-120.0)*0.25;
        if(Z >= 0)
           goto M1170;
        Z=0.0;
M1170:	Y=(98.0-PJ)*(RC2PR-25.0)*0.12;
        if(Y >= 0)
            goto M1190;
        Y=0.0;
/* BO2AD is index of drain oxygenation adequacy and lowers and eventually stops breathing
if too low */
M1190:  U=BO2MT-10.0;   	        // change in brain hypoxia threshold designation GH
        if(U < 0)
            goto M1210;
        U=1.0;
        goto M1220;
M1210:	U=0.0;
M1220:  BO2AD=damp(U,BO2AD,C[63]);
// prevent immediate changes in specified ventilation capacity
        XRESP=damp(C[46],XRESP,C[68]);
// compute total additive effects of vent. stimuli
        U=(C[44]*(X+W)+C[45]*Y+XRESP-Z)*C[47];
// restrict to maximum value, predicted or assumed
        if(U <= C[6])
            goto M1240;
        U=C[6];
/* damp speed of response according to card. output and depth of breathing, incld. brain
oxygenation index */
M1240:	X=(TIDVL+400.0)*C[62]/(COADJ+5.0);
//	SVENT=BO2AD*damp(U,SVENT,X);			        /**/    // appendix IV code
	SVENT=damp(BO2AD*U,SVENT,X);			 // higher stability subsitution MH
        if(PL == GTC)
            goto M1280;
// if vent. stimuli inadequate, = apnoea
	if(SVENT > C[48])
            goto M1290;
	if(NARTI == 0)
            goto M1290;
// P 197
M1280:	DVENT=0;
        RRATE= 0.0001;
        goto M1350;
/* resp.rate calculated from constant + elastance + total ventn.+ art PO2 allowing for
manual adjustment of breathing cap(PR) */
M1290:	if(NARTI == 0)
            goto M1320;
	DVENT=SVENT;
        RRATE=(C[49]+pow(DVENT,0.7)*0.37)*C[50]/(PJ+40.0);
        if(RRATE <= 1.0)
            goto M1350;
        if(COADJ <= 0.5)
            goto M1350;
/* calculate dynamic dead space as sum of anat. factors and those dependent on ventilation 
and perfusion, then calculate alv. ventilation. */
M1320:	U=AO2PR*0.15;
        if(U <= 70.0)
            goto M1340;
        U=70.0;
M1340:	DSPAC=damp((C[52]+DVENT*100.0/pow(RRATE,1.12)+20.0*DVENT/
                   (COADJ+5.0)+U+FY+C[54]*(TIDVL+500.0)),DSPAC,C[55]);
M1350:	TIDVL=DVENT*1000.0/RRATE;
M1360:  AVENT=(TIDVL-DSPAC)*RRATE*FT;
/* restrict tidal volume to maximum(C[20]), then if necessary go back to recompute resp,
rate, providing artifical ventilation not in use  */
        if(NARTI == 0)
            goto M1380;
         X=TIDVL-C[20];
         if(X <= 0)
             goto M1380;
         TIDVL=C[20];
         RRATE=DVENT*1000.0/TIDVL;
         goto M1360;
M1380:	if(AVENT > 0)
	    goto M1400;
        AVENT=0.0;
M1400:	FVENT=AVENT*C[56];
// PG is index of time brain has been deprived of oxygen, if too great death results
        if(BO2AD > 0.3)
            goto M1420;
        PG=PG-(BO2AD-1.0)*C[58];
        goto M_END;
M1420:  PG=PG-BO2AD*C[59];
M_END:  return(item);
        }
// --- estab_pnt() ------------------------------------------------------------------------
// est_pnt() initializes pointers to model() variables for access by other entities
void estab_pnt(void)
    {
    ventilation = &NARTI;
    res_rate    = &RRATE;
    tidal       = &TIDVL;
    positive    = &PEEP;
    bag_index   = &PL;
    bag_vol     = &BAGV;
    bag_co2     = &BAGCP;
    bag_o2      = &BAGOP;
    return;
    }
// --- gases() ----------------------------------------------------------------------------
// P 216
/* gases() takes any values of PO2, PCO2,and PH and determines O2 content and CO2 content
(and saturation of O2) and returns results to calling routine. Maths is that of Kelman
who has published these computer routines. Temperature, haemoglobin and packed cell
volume also allowed for - refer to Kelman's papers for details. */

void gases(float PH, float PO2, float PC2, float *O2CON, float *C2CON, float *SO2)
	{
        float CC, CP, DOX, DR, H, P, PK, SOL, T, X;
 	float Al = -8.532229E3;  float A2 = 2.121401E3;  float A3 = -6.707399E1;
        float A4 = 9.359609E5;   float A5 = -3.134626E4; float A6 = 2.396167E3;
       	float A7 = -6.710441E1;
 	X=PO2*pow(10,(0.4*(PH-DPH)+0.024*(37.0-TEMP)+0.026057669*log(40.0/PC2)));   // VPO2 
     	if(X-0.01 >= 0)
            goto G110;
	X=0.01;
G110: 	if(X-10.0 >= 0)
	    goto G130;
	*SO2 = (0.003683+0.000584*X)*X;
	goto G140;
G130: 	*SO2 = (X*(X*(X*(X+A3)+A2)+Al))/(X*(X*(X*(X+A7)+A6)+A5)+A4);
G140:	*O2CON = HB*(*SO2)*1.34+0.003*PO2;
	P=7.4-PH;
	PK=6.086+P*0.042+(38.0-TEMP)*(0.00472+0.00139*P);
	T=37.0-TEMP;
	SOL = 0.0307+(0.00057+0.00002*T)*T;
	DOX = 0.590+(0.2913-0.0844*P)*P;
	DR = 0.664+(0.2275-0.0938*P)*P;
	T = DOX+(DR-DOX)*(1.0-*SO2);
	CP = SOL*PC2*(1.0+pow(10.0,PH-PK));
	CC=T*CP;
	H=PCV*0.01;
	*C2CON=(CC*H+(1.0-H)*CP)*2.22;
return;
}
// --- gsinv() ----------------------------------------------------------------------------
// P 217 
/* gsinv() reverses gases() and by an optimised process of successive approximation uses O2
and CO2 contents to compute the respective partial pressures
flags ICH1/3 signify states in approximation of PO2, ICH2/4 similarly for PCO2
ICH1=0 means PO2 is well enough approximated with absolute accuracy 'ERR' in O2CON
ICH1=1 means a bracketing procedure to establish bounds on the PO2 value is in progress
ICH3=1 a bracket has been established, a linear interpolation made for next PO2 estimate
ICH3=0 no bracket at this time
set magnitude initial step lengths for bracketing procedure to find bounds within which
PO2 and PCO2 must lie
set initial flag values as if current values for PO2 and PCO2 were as interpolated at a
bracket */
void gsinv(float PH, float *PO2, float *PC2, float *O2CON, float *C2CON, float *SO2)
        {
        float D1,D2,DS,X,XP2,XX1,XX2,Y,XP1,YP1,YP2,YY1,YY2;
        float D1Z=0.5, D2Z=0.5, ERR=0.01;
        int  ICH1=1, ICH2=1, ICH3=1, ICH4=1;                     // partial pressures flags 

// start search from current estimates of PO2 and PCO2
GS100:  gases(PH,*PO2,*PC2,&X,&Y,SO2);
/* XX2/YY2 store discrepancy between known contents and values computed using current
guesses of pressures */
        XX2=X-*O2CON;
        YY2=Y-*C2CON;
// avoid error in sign function below
        if(XX2 < 0) goto GS120;
        if(XX2 == 0) goto GS110;
        if(XX2 > 0) goto GS120;
GS110:  XX2=0.001;
GS120:  if(YY2 < 0) goto GS140;
        if(YY2 == 0) goto GS130;
        if(YY2 > 0) goto GS140;
GS130: YY2=0.001;
/* XP2/YP2 store pressure estimates corresponding to discrepancies XX2/YY2. These are all
used in interpolation procedure below. Form new search vectors D1/D2. After bracketing and
interpolation, according to sign of discrepancies XX2/YY2. Reset ICH3/ICH4 to zero */
GS140:  XP2=*PO2;
        YP2=*PC2;
        if(ICH3-1 < 0) goto GS160;
        if(ICH3-1 == 0) goto GS150;
        if(ICH3-1 > 0) goto GS160;
GS150:  ICH3=0;
        if(-XX2 >= 0)                                                  // Dl=SIGN(D1Z,-XX2)
            D1=fabs(D1Z);
        else
            D1=-fabs(D1Z);
GS160:  if(ICH4-1.0 < 0) goto GS210;
        if(ICH4-1.0 == 0) goto GS170;
        if(ICH4-1.0 > 0) goto GS210;
GS170:  ICH4=0;
        if(-YY2 >= 0)                                                  // D2=SIGN(D2Z,-YY2)
            D2=fabs(D2Z);
        else
            D2=-fabs(D2Z);
/* test for convergence of calculated and given contents. Set flags ICH1/ICH2 to zero if
contents approximated well enough. */
        if(fabs(XX2)-ERR <= 0)
            goto GS180;
        else
            goto GS190;
GS180:  ICH1=0;
GS190:  if(fabs(YY2)-ERR <= 0)
            goto GS200;
        else
            goto GS210;
GS200:  ICH2=0;
// finish if both contents approximated well enough
        if(ICH1+ICH2 <= 0)
            goto GS450;
/* compute trial increment DS for PO2 enforcing a limit of -PO2*0.75 if it is negative.
(Stop negative trial values and smooth approach to a solution.) */
GS210:  DS=D1*ICH1;
        X=*PO2+DS-*PO2*0.25;
        if(X < 0)
            goto GS220;
        else
            goto GS230;
GS220:  DS=-(*PO2*0.75);
GS230:  *PO2=*PO2+DS;
        DS=D2*ICH2; 		         // compute trial increment of PC2 as for PO2 above
        X=*PC2+DS-*PC2*0.25;
// P 218
        if(X < 0)
            goto GS240;
        if(X >= 0)
            goto GS250;
GS240:  DS=-*PC2*0.75;
GS250:  *PC2=*PC2+DS;
        if(*PO2-0.1 < 0)                                  // enforce lower bound on PO2/PC2 
            goto GS260;
        if(*PO2-0.1 >= 0)
            goto GS270;
GS260:  *PO2=0.1;
GS270:  if(*PC2-0.1 < 0)
            goto GS280;
        if(*PC2-0.1 >= 0)
            goto GS290;
GS280:  *PC2=0.1;
// compute contents with trial increments made to PO2/PCO2
GS290:  gases(PH,*PO2,*PC2,&X,&Y,SO2);
/* save last but one pairs of values PO2/O2CON discrepancy in XX1/XP1, similarly for
PC2/C2CON discrepancy in YY1/YP1, place latest pairs of values in XX2/XP2, YY2/YP2 as
before */
        XX1=XX2;
        XX2=X-*O2CON;
        XP1=XP2;
        XP2=*PO2;
        YY1=YY2;
        YY2=Y-*C2CON;
        YP1=YP2;
        YP2=*PC2;
/* first look at discrepancy in O2CON (ICH1=0), if within limit 'ERR' accept values of PO2
and look at C2CON */
        if(fabs(XX2)-ERR <= 0.0)
            goto GS300;
        else
            goto GS310;
GS300:  ICH1=0;
        goto GS360;
// if O2CON discrepancy still too high, test if trial PO2 has overshoot correct solution
GS310:  ICH1=1;
        if(XX2*D1 < 0.0)
            goto GS320;
        if(XX2*D1 == 0.0)
            goto GS360;
        if(XX2*D1 > 0.0)
            goto GS350;
/* If not overshot, solution lies further along direction of trial increment. Compute a
new trial increment based on last 2 pairs of contents/pressures values.*/
GS320:  if(XP2-XP1 == 0 || XX2-XX1 == 0)               // additional divide by zero defence
            goto GS330;
        else
            goto GS340;
GS330:  if(-XX2 >= 0)                                                  // D2=SIGN(D1Z,-XX2)
            D1=fabs(D1Z);
        else
            D1=-fabs(D1Z);
        goto GS360;
GS340: D1=(XP2-XP1)*fabs(XX2)/(fabs(XX2-XX1));
        goto GS360;
/* if overshot, (i.e. bracket found on PO2) linearly interpolate using last 2 pairs of
pressures/contents values to get new trial value for PO2. Set ICH3=1 to indicate bracketed
interpolated value for new search to be initiated at statement 100. Reduce initial search
vector to aid convergence. */
GS350:  ICH3=1;
        *PO2=XP1+(XP2-XP1)*fabs(XX1)/(fabs(XX2)+fabs(XX1));
        D1Z=D1Z/2;
// repeat above procedure for discrepancy in C2CON
GS360:  if(fabs(YY2-ERR) <= 0.0)
             goto GS370;
         else
             goto GS380;
GS370:  ICH2=0;
        goto GS430;
GS380:  ICH2=1;
        if(YY2*D2 < 0)
            goto GS390;
        if(YY2*D2 == 0)
            goto GS430;
        if(YY2*D2 > 0)
            goto GS420;
// keep going
GS390:  if(YP2-YP1 == 0 || YY2-YY1 == 0)               // additional divide by zero defence 
            goto GS400;
        else
            goto GS410;
GS400:  if(-YY2 >= 0)                                                  // D2=SIGN(D2Z,-YY2)
            D2=fabs(D2Z);
        else
            D2=-fabs(D2Z);
        goto GS430;
GS410:  D2=(YP2-YP1)*fabs(YY2)/(fabs(YY2-YY1));
        goto GS430;
  // bracket found 
GS420: ICH4=1;
        *PC2=YP1+(YP2-YP1)*fabs(YY1)/(fabs(YY1)+fabs(YY2));
        D2Z=D2Z/2;
/* Finish if both O2CON and C2CON well enough approximated. If either PO2 or PCO2 search is
at a bracket establish new search at statement GS100. Else if both are continuing along
previous directions of search proceed from statement GS210. */
GS430:  if(ICH1+ICH2 <= 0)
	    goto GS450;
        if(ICH3+ICH4-1 < 0)
            goto GS210;
        else
            goto GS100;
GS450:  return;
        }
// --- constant() -------------------------------------------------------------------------
// constant() computes all parameters constant during a run (P 208)
bool constant(int NREPT)
    {
    float X, EFFIC;
    if(WRATE < 3.0)					       // determine mebabolic rate
        PD=100.0;
    else
        {
        EFFIC=interpol(WRATE, wr_dt, sz_wr_dt);
        PD =(WRATE*1.19*100)/EFFIC;
        }
    C[0] = 0.0;						               // padding, not used
    C[1] = 1000.0/VBLVL;
    C[2] = 100.0/VBLVL;
    C[3] = 0.0203*HB;
    C[4] = (ELAST+105.0)*.01;
    C[5] = 2.7/(VC+0.4);
    C[6] = FEV*25.0+29.0;
    C[7] = CONSO*PD*0.00081*pow(TEMP-26.0,1.05);    // oxygen comsumption index (P 78, 102)
    C[8] = (30.0-PEEP*5.0/ELAST)*0.0016*CONOM*(TEMP-12.2);
    C[9] = (C[7]-CONSO)*0.01; 	         	     // exercise oxygen consumption (P 102)
    C[10] = FT*0.005;
    C[11] = BARPR-0.03*TEMP*TEMP-5.92;      // improved water vapour pressure correction GH
    C[12] = C[11]*0.003592/(273.0+TEMP);	       // pressure conversion factor (P 18)
    C[13] = 0.9/TVOL;
    C[14] = SHUNT+1.0;		      			 // left to right shunt (P 59, 104)
    C[15] = 2.0/WT;
    C[16] = CO*0.01;			      // percentage normal cardiac function (P 103)
    C[17] = FT*10.0;
    C[18] = VADM*80.0;
    C[19] = (PD-90.0)*RVADM*0.05;
    if(C[19] < -1.0)
        C[19] = -1.0;
    C[20] = 650.0*VC;
    if(BULLA >= 0.0)
        C[20] = C[20]+sqrt(BULLA)*15.0;
    C[21] = (40.0-PEEP)*0.025;
    C[22] = 0.0037;                                                           // TC version
    C[23] = 20.0/BARPR;
    C[24] = CONOM*0.3;
    C[25] = 100.*C[12];
    C[26] = 7.0/TVOL;
    C[27] = FT*0.1;
    C[28] = 30000.0/(VBLVL+1000.0);
    C[29] = FT*0.0039*pow(WT,0.425)*pow(HT,0.725);    // catabolism, body size relationship
    C[30] = 520.0/TVOL;
    C[31] = 2.7/TVOL;
    C[32] = C[29]+0.000000l;
    C[33] = C[3]*308.0-TVOL*0.65*C[30];
    C[34] = 0.004/(C[12]*FT);
// P 209 
    C[35] = 0.01/C[12];	        			      // pressure conversion factor
    C[36] = 7.7*C[13];
    C[37] = SPACE/FT;
    C[38] = VLUNG/20.0;                                                       // TC version
    C[39] = FT*0.127;
    C[40] = C[29]*(PD-25.0)*1.3;
    C[41] = FT*(TEMP-24.5)*1.82;   // brain O2 consumption, temperature relationship (P 86)
    C[42] = 0.003*C[29]; 	      			    // related to body surface area
    C[43] = 1.0/(1.0+7.7*C[3]);
    X = pow(PD*0.01,0.8)*VC*0.2;
    C[44] = AZ*X*0.0132;
    C[45] = BZ*X*0.008;
    C[46] = CZ*0.78*(pow(C[7]*0.00051,0.97)+.01);
    X = 0.5+356.0/C[11];
    if(X > 1.0)
        X = 1.0;
    C[47] = PR*0.000214*pow(TEMP-29.0,1.5)*X;
    C[48] = 0.04*(TEMP-26.0)*VC;
    C[49] = 9.0+sqrt(ELAST*1.25);
    C[50] = (150.0+PR)*0.0275*(TEMP-17.0);
    C[51] = RRATE*TIDVL*0.001;
    C[52] = VLUNG*0.03-20.0+BULLA;
    C[53] = 0.02;                                                             // TC version
    C[54] = XDSPA*0.001;
    C[55] = 0.167;                                                            // TC version
    C[56] = 0.001/FT;
    C[57] = FT*60.0;
    C[58] = FT*1.27;
    C[59] = FT*0.3;
    C[60] = FT*0.008;
    X = pow(PD*0.01,1.45); // reduce rate of cardiac ouput increase at start of exercise GH
    if(X > 12.0)
        X = 12.0;
    C[61] = X/(0.015*RCOAJ);                                              // GH, TC version
    C[62] = (CZ+300.0)/4000.0;                                                // TC version
    C[63] = 500/C[7];                                                         // TC version
    C[64] = 0.01488*HB*(TVOL+VBLVL*0.001);
    C[65] = 7.324-CZ*0.00005;
    C[66] = C[65]-0.002;
    C[67] = CZ-30.0;
    C[68] = (PD+200.0)/25.0;  // reduce rate of increase vent exercise start GH, TC version
    C[69] = (FITNS-20.0)*0.00035;
    C[70] = C[29]*1.3;
    C[71] = 21.7+s_sex*1.6;	      // extra constants for 2,3-DPG changes and fitness GH
    C[72] = FT*0.002;
    X=(PD*1.5-150)*0.02-7;
    if(X < -7.0)
        X=7.0;
    if(X > 0.0)
        X=0.0;
    C[73] = FITNS+X;
    ADDC3 = ADDC3/(float)(NREPT);
    DPH = 7.4+(DPG-3.8)*0.025;
    if(DPH > 7.58)
        DPH=7.58;
    if(DPG > 13.0)
        DPG=13.0;
    C[74]=210.0-0.65*AGE;  			         // determine maximum heart rate GH
    C[75]=(1.5+s_sex*0.2+(33.0-FITNS)/10.0)*WT*C[16];	        // maximum stroke volume GH
    return(true);
    }
// P 215
// P 199
// --- bager() ----------------------------------------------------------------------------
// bager() deals with bag rebreathing etc.  It is under development.
void bager(int N, float *CA, float *CB)
{
static int ppl = STD;				    // prior value of the respiration index
float CORR, X;
switch(N)			 // N is a tag that gives the origin of the call to bager()
    {
    case RUN: 				      // source of call, run event ms_resi_main.cpp
	    X=1.0/(BAGV*C[12]);           // set inspired gases equal to mixture in the bag
            FIO2 = BAGO*X;
            FIC2 = BAGC*X;
            if(PL == CDA)                      			     // CO2 absorber fitted
            	FIC2 = 0;                                            		// zero CO2
            break;
    case RSP:                 		     		       // from respiration dialogue
            if(PL == GTC)					       // if glottis closed
    	    	REFLV = VLUNG;                    		    // remember lung volume
	    if((PL == STD) && (ppl == GTC)) // if glottis has just been opened, breathe air
            	{
            	VLUNG = REFLV;              			   // restore all to normal
            	FIO2 = 20.93;         			 // bag stays filled as it was left
            	FIC2 = 0.03;
                }
	    // other RSP situations to be placed here 
            break;
    case MM2:                                         // source of call, model() circa M820
            if(PL == STD || PL == GTC)// DSPVT is gas in upper airway breathed out into bag
                {
                X = DSPAC*RRATE*FT;                 // next section for bag collection only
                BAGV = BAGV+*CA+X;
                X = X*C[12];
                BAGO=BAGO+*CB*AO2MT+FIO2*X;
                BAGC=BAGC+*CB*AC2MT+FIC2*X;
                break;
                }
           if(TIDVL > BAGV)      	 // tidal volume greater than bag volume? yes, exit
                {
                *CB = -10000.0;  	// set up to allow calling function to detect error
                break;
                }
            if(PL != CDA)                 		     // is CO2 absorber in circuit?
                BAGC = BAGC+QB;                             			      // no
            else
                {                                                                    // yes
                BAGV = BAGV-AC2MT*0.01/C[12];
                BAGC = 0;
                }
            BAGO = BAGO-QA;
            BAGV = BAGV-AVENT+*CA;
            break;
    case MM1: 					            // source of call, model() M781
    	    VLUNG = *CA*0.01/C[12];
            break;
    case MM3:// original from deadly() circa D530, here assumed user has set values already
            // X = BARPR-1.2703*TEMP;      		// set BTPS/STPD correction factors
            // CORR=(273.0+TEMP)/(X*0.3592);
            if(BAGV < 0.0)
                BAGV = 0.0;
            // X = X*(273.0 + TEMP)/(273.0*BAGV);
            // BAGP = BAGC*X;
            // BAGPO = BAGO*X;
            // XNMT = BAGV/CORR-BAGO-BAGC;               // should display BAGPO,BAGP,XNMT?
            break;
    case MM4: 					       		      // from deadly() D580
                // X = BARPR-1.2703*TEMP;      	        // set BTPS/STPD correction factors
                // CORR=(273.0+TEMP)/(X*0.3592);
                // X = CORR*100.0/BAGV;
            if(PL == CXA || PL == RBB) 	    // collect expired air in empty bag, P 199 B160
                {
                BAGV = 0.0;
                BAGO = 0.0;
                BAGC = 0.0;
                // XXX = BAGO*X;  // 0.0 ?
                // X = BAGC*X;     // 0.0 ?            	       // should display XXX and X?
                break;
                }
           if(PL == CDA)
                // XXX = BAGO*X;
                // X = BAGC*X;  	   // should display X (as % O2) and XXX (as % CO2)
                break;
    }
ppl = PL;							 // record current PL value
return;
}
// --- death() ----------------------------------------------------------------------------
// death() checks for errors, death and intolerable acidosis.
// It returns true if the subject has died.
bool death(void)
    	{
	if(PG >= 0)
           goto DD150;
	PG=0.0;
DD150: 	if(PG > 7.0)
	   goto DD190;
	if(TPH <= 6.63)
           goto DD210;
	if(TPH > 7.8)
            goto DD230;
	if(BUBBL <= 280.0)
	    goto DD300;
	else
	    goto DD240;
DD190:	strcat(app_buf, "Anoxaemia has been severe and irrecoverable. ");
 	goto DD270;
DD210:	strcat(app_buf, "Tissue pH has fallen to a lethally low level. ");
	goto DD270;
DD230: 	strcat(app_buf, "Tissue pH has risen to a lethally high level. ");
       	goto DD270;
DD240:	strcat(app_buf, "The brain is irrecoverably full of gas bubbles. ");
DD270: 	strcat(app_buf, "I'm sorry, your patient has died.");
	return true;
DD300:	if(PD < 290.0)                    // check for intolerable acidosis during exercise 
	    goto DD360;
	if(TPH > 7.1)
	    goto DD360;
	strcat(app_buf, "... I	can't go on...");
        return true;
DD360:  return false;
        }
// --- sympt() ----------------------------------------------------------------------------
// P 225 
// sympt() specifies various symptoms under appropriate conditions. 
// It returns non-zero if there has been a change in symptoms.
int sympt(void)
        {
// K's and prv_mnr concerned with improvement symptoms (logic is complex)
        int K1=0, K3=0, K5;
        float X;
        prev_state = crrnt_state;
        if(TN2PR > 6000.0)                                 // if tissue N2 partial too high 
            goto S130;
        if(BO2PR < 12.5)                                 // if brain oxygen partial too low 
            goto S130;
        if(BC2PR <= 91.0)
            goto S150;
S130:   if(!tst(S00))
            { strcat(app_buf, "Your patient is unrousable. "); set(S00); }
        K1=1;
        goto S428;
S150:   if(BO2PR > 13.9)
            goto S180;
        if(!tst(S01))
            { strcat(app_buf, "My eyes are going dim. "); set(S01); }
        K1=1;
S180:   if(BUBBL <= 160.1)                         // lowish risk of decompression symptoms
            goto S210;
        if(!tst(S02))
            { strcat(app_buf, "I have awful pains in my legs. "); set(S02); }
        K1=1;
        goto S240;
S210:   if(BUBBL<= 101.1)                           // lower risk of decompression symptoms
            goto S240;
        if(!tst(S03))
            { strcat(app_buf, "My skin is itching. "); set(S03); }
        K3=1;
S240:   if(TN2PR> 4300.0)
            goto S260;
        if(BC2PR <= 80.0)
             goto S280;
S260:   if(!tst(S04))
            { strcat(app_buf, "I am feeling drowsy. "); set(S04); }
        K3=1;
S280:   if(NARTI <= 0)                                          // if artifical ventilation
            goto S390;
        if(RRATE <= 46.0)                                          // if low respirary rate 
            goto S330;
        if(!tst(S05))
            { strcat(app_buf, "This .... is .... imposib... .... "); set(S05); }
S320:   K1=1;
        goto S390;
S330:   if(RRATE <= 35.0)
            goto S360;
        if(!tst(S06))
            { strcat(app_buf, "I am very short of breath. "); set(S06); }
        goto S320;
S360:   if(RRATE <= 27.0)
            goto S390;
        if(!tst(S07))
            { strcat(app_buf, "I am rather short of breath. "); set(S07); }
        K3=1;
S390:   if(TPH <= 7.59)                                                        // tissue pH
            goto S420;
        if(!tst(S08))
           { strcat(app_buf, "I am getting tingling and cramps in my hands. "); set(S08); }
        K3=1;
S420:   if(TPH > 7.13)                                                         // tissue pH
            goto S428;
        if(!tst(S09))
            { strcat(app_buf, "I don't feel well at all. "); set(S09); }
        K3=1;
S428:   if(TPH >= 7.08)                                                        // tissue pH 
            goto S450;
        if(!tst(S10))
            { strcat(app_buf, "Your patient is twitching. "); set(S10); }
             K1=1;
S450:   if(NARTI > 0)                                             // if natural ventilation
            goto S490;
        X=SVENT*0.9-DVENT;
        if(X <= 0)
            goto S490;
        if(!tst(S11))
            { strcat(app_buf, "Your patient is fighting the ventilator. "); set(S11); }
        K3=1;
// P 226
S490:   X=1.3*HB-RO2CT;
        if(X < 7.0)
            goto S520;
        if(!tst(S12))
            { strcat(app_buf, "Your patient is very blue. "); set(S12); }
        K1=1;
        goto S550;
S520:   if(X < 5.0)
            goto S550;
        if(!tst(S13))
            { strcat(app_buf, "Your patient is blue. "); set(S13); }
        K3=1;
S550:   if(K1 <= K3)
            goto S570;
        K3=1;
S570:   K5=K1+K3;
        if(prv_svr-K5 <= 0)
            goto S600;
        strcat(app_buf,
                  "God bless you doctor. I feel really well again. It's like a miracle. ");
        clr(ALL);
            goto S660;
S600:   if(prv_svr <= K1)
            goto S630;
        strcat(app_buf,
        	  "I feel better but not right yet. Can't you do something else for me? ");
        clr(ALL);
        goto S660;
S630:   if(prv_mnr <= K3)
            goto S660;
        strcat(app_buf,
      "That's better doctor. ...  But are you going to do any other nasty things to me? ");
        clr(ALL);
S660:   prv_svr-=K1;
        prv_mnr=K3;
        return(prev_state - crrnt_state);
        }
// --- interpol() -------------------------------------------------------------------------
// interpol() interpolates in table of coordinates to give y value out for x value in.  It
// has been modelled on GH FUNCTN(EN, OU, FUN, NBP)
float interpol(float in, struct coord *pcd, int sz)
{
int i;
if(in <= pcd->x)           					          // x below range?
     return(pcd->y);
if(in >= (pcd+sz-1)->x) 					          // x above range?
     return((pcd+sz-1)->y);
for(i = 1; i < sz; i++)
     if((pcd+i)->x > in)
        break;
return
((pcd+i-1)->y + (((pcd+i)->y-(pcd+i-1)->y)/((pcd+i)->x-(pcd+i-1)->x)) * (in-(pcd+i-1)->x));
}
// --- damp() -----------------------------------------------------------------------------
// damp() implements a single pole filter.  It takes the input value of the variable, the
// current value of the variable, the time constant in iteration periods and returns the
// output value for that iteration period.  It is scaled in seconds.
// damp() should be appropriately scaled if the iteration period is changed from one second
float damp(float in, float crnt, float tc)
    {
    tc = tc < 0.0? 0.0: tc;                                    // time constant must be +ve
    tc = tc > 120.0? 120.0: tc;      // limit time constant to physological possible values
    return(in/(tc+1) + tc*crnt/(tc+1));
    }
// --- delay() ----------------------------------------------------------------------------
// P 215
// delay() The delay is obtained by stepping back into history.  The stepback is related
// to the effective cardiac output as determined by dly_dt[].
void delay(float coadj)
{
    static int index;                                     // current position in delay line
    int sb;                       // stepback into history of variables, that is, the delay
    if(coadj < 0.0)                                     		  // initialization
    	{
        for(index = 0; index < NUM_DLY; index++)      // preload delay with sensible values
            {
            TDLAY[index][0] = VO2CT;
            TDLAY[index][1] = VC2CT;
            TDLAY[index][2] = VC3MT;
            TDLAY[index][3] = TC2PR;
            }
        index = 0;
        }
   else                                                     		 // do delay action
        {
        TDLAY[index][0] = VO2CT;                   // load current values into delay buffer
        TDLAY[index][1] = VC2CT;
        TDLAY[index][2] = VC3MT;
        TDLAY[index][3] = TC2PR;
        sb = interpol(coadj, dly_dt, sz_dly_dt)+0.5;		      // determine stepback
        sb = index-sb;							// indexed stepback
    	if(sb < 0)							 // catch rollunder
            sb += NUM_DLY;
        VO2CT = TDLAY[sb][0];                          		  // pick up earlier values
        VC2CT = TDLAY[sb][1];
        VC3MT = TDLAY[sb][2];
        TC2PR = TDLAY[sb][3];
        index++;							// move up one step
	if(index >= NUM_DLY)		     			          // catch rollover
    	    index -= NUM_DLY;
        }
    return;
}
// --- load_new() -------------------------------------------------------------------------
// loads untainted patient parameters
bool load_new(void)
    {
    int i;
    for(i = 0; i < NUM_FCT; i++)
    	factor[i].vlu = s_factor[i];	 		  	            // load factors
    for(i = 0; i < NUM_DSY-1; i++)
    	display[i].vlu = s_display[i].fresh; ;			 // load display parameters
    for(i = 0; i < NUM_OTH; i++)                                   // load other parameters
        other[i] = s_other[i];
    HT = s_height;
    WT = s_weight;
    AGE = s_age;
    sex = s_sex;
    prv_mnr = 0;               		               // zero previous symptoms indicators
    prv_svr = 0;
    clr(ALL);                                            // clear all clinical message bits
    delay(-1);								// prime delay line
    return(true);
    }
// --- xxxxxxxxx --------------------------------------------------------------------------

