// --- ms_resi_main.cpp -------------------------------------------------------------------

// Ms Resi was developed from MacPuf. MacPuf is detailed in
// "A Computer Model of Human Respiration", C. J.Dickinson, MTP Press, 1977.
// Further text on authorship is available in 'ms_string.h'.
// Mal Haysom, Civil Engineering and Physical Sciences, La Trobe University, December 2008

// --- version history --------------------------------------------------------------------
// (VERSION is defined in ms_string.h)
// V 1.00 introductory version
// 19/05/2009 errors spotted by Michelle Gibson corrected
// circa 21/05/2009 V 1.10 posted
// 24/05/2009 transient on re-run corrected
// 24/05/2009 V1.11 posted
// 01/06/2009 delay() look-up table structured
// 04/06/2009 'minor' changes from Version 17 12/1981 George Havenith included, noted as GH
// 11/07/2009 WRATE introduced PD/WRATE relationship as per GH
// 11/07/2009 GH catabolism and TRQ code included, delay() implemented with interpol()
// 11/07/2009 subject file format changed - tagged as #001
// 17/07/2009 V1.20 posted
// 30/07/2009 print facility implemented
// 30/07/2009 V1.21 posted
// 10/08/2009 print facility B&W printer friendly, clinical comments included
// 17/08/2009 time display on clinical comments improved
// 02/09/2009 damp() macro replaced by time constant based damp() function (noted as TC)
// 02/09/2009 change at M1241 to improve stability at low oxygen levels
// 08/09/2009 V1.23 posted
// 26/09/2009 death parameters system introduced, jog system on cursor introduced
// 26/09/2009 error in initialization of C[71] corrected
// 26/09/2009 BMI introduced, print command moved to File menu
// 27/09/2009 print bug that occured with empty clinical display rectified
// 29/09/2009 V1.24 posted
// 07/12/2009 V2.00 interface/model system posted
// 09/11/2012 V2.01 cosmetic corrections to cursor placement
// ----------------------------------------------------------------------------------------
#include <vcl.h>
#include <fstream.h>                                 
#include <math.h>
#include <string.h>
#include <io.h>
#include <dir.h>
#include <stdio.h>
#include <printers.hpp>
#include <DateUtils.hpp>
#pragma hdrstop

#include "ms_resi_main.h"
#include "ms_string.h"      // string constants for the introduction card and the about box
#include "ms_variable.h"             // variable structure definition and utility functions
#include "about.h"
#include "respiration.h"

// ----------------------------------------------------------------------------------------
#pragma package(smart_init)
#pragma resource "*.dfm"

// --- chart, plot defines ----------------------------------------------------------------
#define CL      2                                                     // left edge of chart
#define CR      (main_F->ClientWidth)                                        // chart right
#define CH      (383)                                                       // chart height
#define GL      (CL+SM-1)                                                      // plot left
#define GR      (CR-SM)                                                       // plot right
#define GH      (CH-(TM+BM))                                                 // plot height
#define SM      160                                                         // side margins
#define BM      30                                                         // bottom margin
#define TM      20                                                            // top margin
#define CLB_SPC	38                                             // calibration scale spacing
#define NREPT	60					        // length of run in seconds

// --- print defines ----------------------------------------------------------------------
#define MM      (data_P->PageHeight/202.0)                                // approximate mm
#define PVA     48.0                                   // print vertical axis placement, mm
#define PCS     16.0                                                // print column spacing
#define NMC     12.0                                             // number of print columns
#define CCS     11.0                                          // calibration column spacing
#define PHA     120.0                                    // print horizontal axis placement
#define PHT     100.0                                                       // print height
#define PSC     (PCS/60.0)                                              // print second, mm

// --- introduction card defines ----------------------------------------------------------
#define IW      430                                              // introduction card width
#define IH      315                                             // introduction card height
#define IL      275                                               // introduction card left
#define IT      25                                                 // introduction card top

// --- utility defines --------------------------------------------------------------------
#define BFA_SZE 5000                                           // for larger working buffer
#define BFB_SZE 1000                                          // for smaller working buffer
#define NUM_ITM	8	       	        // number of items in each of the two select panels
#define MK_CNV  0.1332                                 // pressure conversion, mm Hg to kPa

// --- utility functions ------------------------------------------------------------------
int run(void);     // interacts with the core model software to do 60 seconds of simulation
bool update(void); 					   // update (re)draws display area
bool update_pnls(int second); 	      // updates factor and display panels values and units
bool calibrate(void);		 // provides calibration scales on either side of plot area
bool intro(bool show);                                        // displays introduction card
char *format(float value, char *f_s);              // formats value for calibration display
int plot(void); // plots current display variables, returns displacement of last plot point
bool draw_cursor(int cursor); 			    		       // draws cursor line
bool new_subject(void); 			 // performs loads and resets for fresh run
bool limit(variable *ppp, TEdit *pv_E, char *format = "%.2f");    // adjusts data to limits
void trad_metric(bool kpa);    // loads items from alternate lists, does presure conversion
float metric_conv(int v, int t);      // if applicable converts display variables to metric
int satof(char *p, float *v, int max);   // loads float value and returns pointer increment
void load_hints(void);                         // sets hint messages for panels and buttons
void alert(int message, int bell = 0);               // displays alert message, audio alert
float mark(int itm, float);            // check for overlap of chart markers, return offset
bool item(int item);				      // alerts user to non-physical values
void append(void); 	                      	       // appends messages to clinical memo
int wrap(int index); // takes index to history buffer and if necessary adjusts for wrapping

// --- imported variables -----------------------------------------------------------------
trade char *comment;                                                // ms_model information
trade char *mdl_ver;                                             // ms_model version number
trade variable factor[], display[];                            // factor and display arrays
trade int *bag_index;                               // pointer to index required by bager()
trade float other[];                                               // array of other values
trade float history[/* variable index */][HST_LNG];   // history of physiological variables
trade variable alt_fact[];                                    // alternate factor variables
trade variable alt_dspy[];                                   // alternate display variables
trade int a_f_lst[];                                    // array of alterate factor indexes
trade a_d_lst[];                                      // array of alternate display indexes
trade float s_factor[], s_other[];  		 // factor and other initial default values
trade struct s_dsply s_display[];		// display initial and final default values
trade variable height, weight, age;                                  // passport parameters
trade int sex;                                  // passport parameter, 0 = female, 1 = male
trade float s_age, s_height, s_weight;                      // template passport parameters
trade int s_sex;					     // template passport parameter
trade int num_fct;
trade int num_dsy;
trade int num_oth;
trade char app_buf[];					         // clinical message buffer

// --- time related variables -------------------------------------------------------------
int second; 					        // elapsed time since fresh subject
int int_mnt;	// integer maximum number of minutes in plot time axis expressed in seconds
int cursor;						    // placement of cursor, seconds
int start;                                                        // start of plot, seconds
int second_max;							       // maximum excursion

// --- utility variables ------------------------------------------------------------------
bool intro_f;              	                 // determines display of introduction card
bool cursor_f;    	                                     // placement of cursor by user
bool metric;                                // pressure units flag, true: kPa, false: mm Hg
enum TColor d_clr[] = {clBlack,clRed,clGreen,clBlue,clFuchsia,clOlive,clTeal,clGray};
char wrk_bfa[BFA_SZE];                                               // general work buffer
char wrk_bfb[BFB_SZE];                                               // smaller work buffer

struct 	var_st 	{                                        // structure of each variable item
		TComboBox *dsn;
		TEdit     *vlu;
		TLabel    *unit;
        	} ;

var_st fact[NUM_ITM];                                           // subject factor variables
var_st dspy[NUM_ITM];                                                  // display variables
int crnt_fact[NUM_ITM] = {16, 15, 5, 0, 32, -1, -1, -1};// current subject factor variables
int crnt_dspy[NUM_ITM] = {15, 10, 40, 52, 36, 35, 63, 63};     // current display variables
char crnt_sbj[100] = {'\0'};                                   // current subject file name

// --- main -------------------------------------------------------------------------------
Tmain_F *main_F;

// ----------------------------------------------------------------------------------------
__fastcall Tmain_F::Tmain_F(TComponent* Owner) : TForm(Owner)
    {
    load_hints();
    estab_pnt(); 			      // initialize pointers to model.cpp variables
    metric_RB->Checked = false;                          // default to mm Hg pressure units
    trad_RB->Checked = true;
    metric = false;
    new_subject();
    }
// ----------------------------------------------------------------------------------------
void __fastcall Tmain_F::FormPaint(TObject *Sender)
    {
    update();
    }
// --- main_F create ----------------------------------------------------------------------
void __fastcall Tmain_F::FormCreate(TObject *Sender)
{
int i, j;
fact[0].dsn = f_dsn_CB0; fact[0].vlu = f_vlu_E0; fact[0].unit = f_unit_L0;  // factor panel
fact[1].dsn = f_dsn_CB1; fact[1].vlu = f_vlu_E1; fact[1].unit = f_unit_L1;
fact[2].dsn = f_dsn_CB2; fact[2].vlu = f_vlu_E2; fact[2].unit = f_unit_L2;
fact[3].dsn = f_dsn_CB3; fact[3].vlu = f_vlu_E3; fact[3].unit = f_unit_L3;
fact[4].dsn = f_dsn_CB4; fact[4].vlu = f_vlu_E4; fact[4].unit = f_unit_L4;
fact[5].dsn = f_dsn_CB5; fact[5].vlu = f_vlu_E5; fact[5].unit = f_unit_L5;
fact[6].dsn = f_dsn_CB6; fact[6].vlu = f_vlu_E6; fact[6].unit = f_unit_L6;
fact[7].dsn = f_dsn_CB7; fact[7].vlu = f_vlu_E7; fact[7].unit = f_unit_L7;

dspy[0].dsn = d_dsn_CB0; dspy[0].vlu = d_vlu_E0; dspy[0].unit = d_unit_L0; // display panel
dspy[1].dsn = d_dsn_CB1; dspy[1].vlu = d_vlu_E1; dspy[1].unit = d_unit_L1;
dspy[2].dsn = d_dsn_CB2; dspy[2].vlu = d_vlu_E2; dspy[2].unit = d_unit_L2;
dspy[3].dsn = d_dsn_CB3; dspy[3].vlu = d_vlu_E3; dspy[3].unit = d_unit_L3;
dspy[4].dsn = d_dsn_CB4; dspy[4].vlu = d_vlu_E4; dspy[4].unit = d_unit_L4;
dspy[5].dsn = d_dsn_CB5; dspy[5].vlu = d_vlu_E5; dspy[5].unit = d_unit_L5;
dspy[6].dsn = d_dsn_CB6; dspy[6].vlu = d_vlu_E6; dspy[6].unit = d_unit_L6;
dspy[7].dsn = d_dsn_CB7; dspy[7].vlu = d_vlu_E7; dspy[7].unit = d_unit_L7;

d_vlu_E0->Font->Color = d_clr[0];      d_vlu_E1->Font->Color = d_clr[1]; // display colours
d_vlu_E2->Font->Color = d_clr[2];      d_vlu_E3->Font->Color = d_clr[3];
d_vlu_E4->Font->Color = d_clr[4];      d_vlu_E5->Font->Color = d_clr[5];
d_vlu_E6->Font->Color = d_clr[6];      d_vlu_E7->Font->Color = d_clr[7];

for(i = 0; i < NUM_ITM; i++)                                        // for each factor item
    {
     for(j = 0; j < num_fct; j++)                                        // for each factor
         fact[i].dsn->Items->Add(factor[j].dsn);
     }

for(i = 0; i < NUM_ITM; i++)                                       // for each display item
    {
     for(j = 0; j < num_dsy; j++)                              // for each display variable
         dspy[i].dsn->Items->Add(display[j].dsn);
      }
for(i = 0; i < NUM_ITM; i++)					      // load current items
    {
    fact[i].dsn->ItemIndex = crnt_fact[i];
    dspy[i].dsn->ItemIndex = crnt_dspy[i];
    }
update_pnls(0);
}
// --- factor or display item change ------------------------------------------------------
void __fastcall Tmain_F::dsn_CBChange(TObject *Sender)
    {
    int i, j;
    bool match;
    for(i = 0; i < NUM_ITM; i++)
    if(Sender == fact[i].dsn)                         // factors not permitted to double up
        {
        match = false;
        for(j = 0; j < NUM_ITM; j++)      // permit only one example of each factor on list
             if(fact[i].dsn->ItemIndex == crnt_fact[j] && i != j)   // don't count yourself
                 {
                 crnt_fact[i] = -1;
                 match = true;
                 alert(1, 1);						       // tell user
                 }
        if(!match)
           crnt_fact[i] = fact[i].dsn->ItemIndex;
        }
     else
        crnt_dspy[i] = dspy[i].dsn->ItemIndex;

    cursor_f = true;   						// maintain cursor position
    update_pnls(cursor);
    if(second != 0)                                                     // if plot existent
        update();                                                    // plot new data items
    }
// --- factor_I click ---------------------------------------------------------------------
// displays list of subject factors.
void __fastcall Tmain_F::factor_IClick(TObject *Sender)
{
int i;
strcpy (wrk_bfa, "Subject Factor List\r\n\r\n    ");
for(i = 0; i < num_fct; i++)
    {
    strcat(wrk_bfa, factor[i].dsn);
    strcat(wrk_bfa,  "\r\n    ");
    }
about_F->Caption = "Ms Resi Subject Factors";
about_F->text_M->Height = 14*num_fct+9;
about_F->text_M->ScrollBars = ssNone;
about_F->okay_B->Top = 14*num_fct+142;
about_F->Position = poMainFormCenter;
about_F->Height = 14*num_fct+202;
about_F->version_L->Caption = "V: " VERSION "   "__DATE__;
about_F->text_M->Text = wrk_bfa;
about_F->ShowModal();
}
// --- display_I click --------------------------------------------------------------------
// displays list of display variables.
void __fastcall Tmain_F::display_IClick(TObject *Sender)
{
int i;
strcpy (wrk_bfa, "Display Variables List\r\n");
for(i = 0; i < num_dsy-1; i++)            		// last display variable is a dummy
    {
    strcat(wrk_bfa,  "\r\n    ");
    strcat(wrk_bfa, display[i].dsn);
    }
about_F->Caption = "Ms Resi Display Variables";
about_F->text_M->Height = 479;
about_F->text_M->ScrollBars = ssVertical;
about_F->okay_B->Top = 612;
about_F->Position = poMainFormCenter;
about_F->Height = 672;
about_F->version_L->Caption = "V: " VERSION "   "__DATE__;
about_F->text_M->Text = wrk_bfa;
about_F->ShowModal();
}
// ---- exit_I click ----------------------------------------------------------------------
void __fastcall Tmain_F::exit_IClick(TObject *Sender)
    {
    exit(0);
    }
// ---- about_I click ---------------------------------------------------------------------
void __fastcall Tmain_F::about_IClick(TObject *Sender)
{
about_F->Caption = "Ms Resi  About";
about_F->text_M->Height = 489;
about_F->text_M->ScrollBars = ssNone;
about_F->okay_B->Top = 616;
about_F->Position = poMainFormCenter;
about_F->Height = 682;
about_F->version_L->Caption = "V: " VERSION "   "__DATE__;
about_F->text_M->Text = authors;
about_F->ShowModal();
}
// --- run_B click ------------------------------------------------------------------------
void __fastcall Tmain_F::run_BClick(TObject *Sender)
    {
    run();                  // interact with the core model to run 60 seconds of simulation
    }
// --- reset_B click ----------------------------------------------------------------------
void __fastcall Tmain_F::reset_BClick(TObject *Sender)
    {
    new_subject();
    update();
    }
// --- metric_RB click --------------------------------------------------------------------
void __fastcall Tmain_F::metric_RBClick(TObject *Sender)
    {
    metric_RB->Checked = true;
    trad_RB->Checked = false;
    metric = true;
    intro_f = false;
    trad_metric(true);
    update();
    }
// --- trad_RB click ----------------------------------------------------------------------
void __fastcall Tmain_F::trad_RBClick(TObject *Sender)
    {
    metric_RB->Checked = false;
    trad_RB->Checked = true;
    intro_f = false;
    metric = false;
    trad_metric(false);
    update();
    }
// --- main_F mouse down ------------------------------------------------------------------
// establish cursor position
void __fastcall Tmain_F::FormMouseDown(TObject *Sender,
                                      TMouseButton Button, TShiftState Shift, int x, int y)
    {
    if(Button != mbLeft)                                      // deal with left clicks only
   	return;
     x -= SM;                                                 	   // refer to left of plot
     if((x > 0 && x <= second-start) && (y > TM && y < CH))               // if valid point
    	{
    	cursor = x + start; 						    // in real time
    	cursor_f = true;
    	update();
    	}
    else
       alert(3, 1);              				     // alert user to error
    return;
 }
// --- alt/ctrl left/right arrows and alt I -----------------------------------------------
void __fastcall Tmain_F::FormKeyDown(TObject *Sender, WORD &Key, TShiftState Shift)
    {
    int i;
/*    if(Shift.Contains(ssCtrl) && (Key == 'R'))           // run for 5 minutes on ctrl + r
        {
        for(i = 0; i < 5; i++)       trouble can occurs if changing factor
            run();
        return;
        }         */
    if((Key == VK_LEFT || Key == VK_RIGHT) && Shift.Contains(ssAlt))        // cursor shift
        {
        if(!Shift.Contains(ssShift))
            {
            if(Key == VK_LEFT)						// alt + left arrow
                cursor > start? cursor--: start;
            else						       // alt + right arrow
                cursor < second? cursor++: second;
            }
        if(Shift.Contains(ssShift))
            {
            if(Key == VK_LEFT)					// alt + shift + left arrow
               for(i = 0; (i < 60) && (cursor > start); i++)
                   cursor--;
            else					       // alt + shift + right arrow
               for(i = 0; (i < 60) && (cursor < second); i++)
                   cursor++;
            }
        cursor_f = true;
        update();
        }
    if(Shift.Contains(ssAlt) && (Key == 'I'))                         // information window
    	MessageDlg(comment, mtInformation, TMsgDlgButtons() << mbOK, 0);  
    }
// --- passport exit ----------------------------------------------------------------------
// updates changed values in passport panel
void __fastcall Tmain_F::passportExit(TObject *Sender)
    {
    limit(&height, height_E, "%.1f");
    limit(&weight, weight_E, "%.1f");
    sprintf(wrk_bfa, "%.1f", 10000.0*weight.vlu/(height.vlu*height.vlu));   // evaluate bmi
    bmi_E->Text = wrk_bfa;
    limit(&age, age_E, "%.1f");
    sex = sex_CB->ItemIndex;
    }
// ---- factor exit -----------------------------------------------------------------------
void __fastcall Tmain_F::factorExit(TObject *Sender)
    {
    int i, j;
    for(i = 0; i < NUM_ITM; i++)
    	{
        j = crnt_fact[i];
        if(j != -1)        						   // "select" item
    	    limit(&factor[j], fact[i].vlu);
        else
            fact[i].vlu->Text = " - - - - -";
        }
    }
// --- back_B click -----------------------------------------------------------------------
void __fastcall Tmain_F::back_BClick(TObject *Sender)
    {
    cursor_f = true;                                                        /**/
    start -= 60;
    if(start < 0)    						    // reject negative time
    	{
    	start = 0;
    	alert(4, 1);							      // alert user
    	}
    if(start < second_max - HST_LNG)   			  // history is limited by wrapping
    	{
    	start = second_max - HST_LNG;
    	alert(5, 1);                                          	     // alert user to limit
    	}
    update();
    }
// --- forward_B click --------------------------------------------------------------------
void __fastcall Tmain_F::forward_BClick(TObject *Sender)
    {
    start += 60;
    if(start > second)						      // nothing to display
       	{
    	start = second;                                                  	// stagnate
    	alert(6, 1);  							      // alert user
    	}
    update();
    }
// --- rerun_B click ----------------------------------------------------------------------
void __fastcall Tmain_F::rerun_BClick(TObject *Sender)
    {
    int i;
    second = cursor;
    for(i = 0; i < num_dsy; i++)	     	 // load conditions existent at cursor time
    	display[i].vlu = history[i][wrap(second)];
    for(i; i < num_dsy+num_oth; i++)
        other[i-num_dsy] = history[i][wrap(second)];
    update();
    }
// --- print_I click ----------------------------------------------------------------------
// prints current screen data
void __fastcall Tmain_F::print_IClick(TObject *Sender)
{
int i, j, k, m, n, h_os;                                               // horizontal offset
float lower, upper, scale, os, x, y;
TRect frame;
char f_s[10];
TPrinter *data_P = Printer();
data_P->Orientation = poLandscape;
if(print_DB->Execute())
    {
    frame = Rect(0,0,data_P->PageWidth,data_P->PageHeight);
    data_P->BeginDoc();
    data_P->Canvas->Brush->Color = clBlack;
    data_P->Canvas->Font->Color = clBlack;                           // default text colour
    data_P->Canvas->Pen->Color = clBlack;
// - - - outer frame and title - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    data_P->Canvas->FrameRect(frame);
    data_P->Canvas->Brush->Color = clWhite;
    data_P->Canvas->Font->Size = 12;                                      // for title text
    sprintf(wrk_bfa, "Ms Resi    %s  %s", crnt_sbj, DateTimeToStr(Now()).c_str());
    data_P->Canvas->TextOutA(10*MM, 2*MM, wrk_bfa);
    data_P->Canvas->Font->Size = 10;
// - - - horizontial grid - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    for(i = 0; i < 11; i++)                                       // 10 divisions, 11 lines
        {
        data_P->Canvas->MoveTo(PVA*MM, (PHA-i*10)*MM);
        data_P->Canvas->LineTo((PVA+NMC*PCS)*MM, (PHA-i*10)*MM);
        }
// - - - vertical grid, minute ticks, minute text - - - - - - - - - - - - - - - - - - - - -
    for(i = 0; i <= NMC*60; i+=15)               // vertical grid (15 second, 4 mm spacing)
        {
        if(!(i%60))                                                       // integer minute
            {
            data_P->Canvas->MoveTo((PVA+i*PSC)*MM, PHA*MM);
            data_P->Canvas->LineTo((PVA+i*PSC)*MM, (PHA-PHT)*MM);
            sprintf(wrk_bfa, "%d", (start+i)/60);
            data_P->Canvas->TextOutA((PVA-2+i*PSC)*MM, (PHA+1)*MM, wrk_bfa); // minute text
            }
            if(!((i+30)%4))                                            // half-minute ticks
            {
            data_P->Canvas->MoveTo((PVA+i*PSC)*MM, PHA*MM);
            data_P->Canvas->LineTo((PVA+i*PSC)*MM, (PHA-3)*MM);
            }
        if(!((i+15)%2))                                             // quarter-minute ticks
            {
            data_P->Canvas->MoveTo((PVA+i*PSC)*MM, PHA*MM);
            data_P->Canvas->LineTo((PVA+i*PSC)*MM, (PHA-1.5)*MM);
            }
        }
    data_P->Canvas->TextOutA((PVA+3+NMC*PCS)*MM, (PHA+1)*MM, "minutes");  // minute caption
// - - - calibration scales - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    h_os = 4;                                                          // left items offset
    for(i = 0; i < NUM_ITM; i++)                              // display items calibrations
    {
    if(i == 4)
        h_os = (PCS*NMC+8);                                           // right items offset
    for(j = 0; j < 11; j++)
         {
         lower = display[dspy[i].dsn->ItemIndex].lwr;
         upper = display[dspy[i].dsn->ItemIndex].upr;
         if((upper-lower) != 0)
             {
             data_P->Canvas->Pen->Color = d_clr[i];                          // line colour
             data_P->Canvas->MoveTo((h_os+i*CCS)*MM, PHA*MM);     // draw axis, from bottom
             data_P->Canvas->LineTo((h_os+i*CCS)*MM, (PHA-PHT)*MM);               // to top
             sprintf(wrk_bfa, format(upper, f_s), lower + j*0.1*(upper-lower));
             data_P->Canvas->Font->Color = d_clr[i];                         // text colour
             data_P->Canvas->TextOutA((h_os+1+i*CCS)*MM, (PHA-3-j*10)*MM, wrk_bfa);
             sprintf(wrk_bfa, "(%d)", i+1);                       // numerically label axis
             data_P->Canvas->TextOutA((h_os+2+i*CCS)*MM, (PHA-PHT-7)*MM, wrk_bfa);
             }
         }
    }
// - - - draw cursor - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    k = cursor-start;
    data_P->Canvas->Pen->Color = clRed;
    data_P->Canvas->Pen->Style = psDash;
    data_P->Canvas->MoveTo((PVA+k*PSC)*MM, PHA*MM);           	   	   // main vertical
    data_P->Canvas->LineTo((PVA+k*PSC)*MM, (PHA-PHT)*MM);
    data_P->Canvas->Pen->Style = psSolid;
    data_P->Canvas->Pen->Width = 0.5*MM;
    data_P->Canvas->MoveTo((PVA+k*PSC-1.5)*MM, (PHA-PHT-2.5)*MM);               // top tick
    data_P->Canvas->LineTo((PVA+k*PSC)*MM, (PHA-PHT-0.5)*MM);
    data_P->Canvas->LineTo((PVA+k*PSC+1.5)*MM, (PHA-PHT-2.5)*MM);
    data_P->Canvas->MoveTo((PVA+k*PSC-1.5)*MM, (PHA+7)*MM);                  // bottom tick
    data_P->Canvas->LineTo((PVA+k*PSC)*MM, (PHA+5)*MM);
    data_P->Canvas->LineTo((PVA+k*PSC+1.5)*MM, (PHA+7)*MM);
    data_P->Canvas->Pen->Width = 1;
// - - - data plots - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    mark(-1, 0.0);                                                 // clear list of markers
    data_P->Canvas->Pen->Width = 0.3*MM;   // broaden plot (required for monotone printers)
    for(i = 0; i < NUM_ITM; i++)      		     // form plots of current display items
      	{
        j = crnt_dspy[i];                                       // for notation convenience
        if(display[j].lwr == display[j].upr)
            continue;					   // no display item or data error
       	data_P->Canvas->Pen->Color = d_clr[i];                	    	    // colour coded
	scale = PHT/(display[j].lwr - display[j].upr);   // print coordinates +ve downwards
      	os = PHA - scale*display[j].lwr;                                 // vertical offset
      	for(k = 0; k < second-start; k++) 			     // for each plot point
            {
            if((k*PSC) >= NMC*PCS) // restrict plot to plot area (after use of back button)
            	break;
            m = wrap(start+k);
            n = wrap(start+k+1);
            if(metric_conv(j, n) < display[j].upr && metric_conv(j, n) > display[j].lwr)
                {
                data_P->Canvas->MoveTo((PVA+k*PSC)*MM, (metric_conv(j,m)*scale+os)*MM);
                data_P->Canvas->LineTo((PVA+(k+1)*PSC)*MM, (metric_conv(j,n)*scale+os)*MM);
                }
            }
        x = PVA+(cursor-start)*PSC;
        y = metric_conv(j, cursor)*scale+os;
        x += mark(i, y);
        data_P->Canvas->Ellipse((x-2)*MM, (y-2)*MM, (x+2)*MM, (y+2)*MM);   // place markers
        data_P->Canvas->TextOutA((x-1)*MM, (y-2)*MM, itoa(i+1, wrk_bfa, 10));
        }
    data_P->Canvas->Pen->Width = 1;                          // back to minimum width lines
// - - - clinical - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    data_P->Canvas->Font->Color = clRed;
    sprintf(wrk_bfa, "%s", main_F->clinical_M->Text);
     j = 0;
     for(i = 0; i < 4; i++)              // yes, I know there is a proper way of doing this
        {
        if(wrk_bfa[j] == '\0' || wrk_bfa[6] == '\0')    // be alert to wrk_bfa = "(null)\0"
            break;
        for(k = 0; wrk_bfa[j] != '\r'; k++)
            wrk_bfb[k] = wrk_bfa[j++];
        wrk_bfb[k] = '\0';
        data_P->Canvas->TextOutA(33.0*MM, (PHA+9.0+5.0*i)*MM, wrk_bfb);       // print out
        j += 2;                                                         // step over "\r\n"
       }
// - - - passport data - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    data_P->Canvas->Font->Color = clBlack;
    data_P->Canvas->TextOutA(5.0*MM, (PHA+40.0)*MM, "Height");               // height text
    sprintf(wrk_bfa, "%.1f %s", height.vlu, height.unt);
    data_P->Canvas->TextOutA(20.0*MM, (PHA+40.0)*MM, wrk_bfa);
    data_P->Canvas->TextOutA(5.0*MM, (PHA+45.0)*MM, "Weight");               // weight text
    sprintf(wrk_bfa, "%.1f %s", weight.vlu, weight.unt);
    data_P->Canvas->TextOutA(20.0*MM, (PHA+45.0)*MM, wrk_bfa);
    data_P->Canvas->TextOutA(5.0*MM, (PHA+50.0)*MM, "Age");                     // age text
    sprintf(wrk_bfa, "%.0f %s", age.vlu, age.unt);
    data_P->Canvas->TextOutA(20.0*MM, (PHA+50.0)*MM, wrk_bfa);
    data_P->Canvas->TextOutA(5.0*MM, (PHA+55.0)*MM, "Sex");                     // sex text
    sprintf(wrk_bfa, "%s", sex?"male":"female");
    data_P->Canvas->TextOutA(20.0*MM, (PHA+55.0)*MM, wrk_bfa);
// - - - subject factors - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    for(i = 0; i < NUM_ITM; i++)                                           // display items
        {
        if(crnt_fact[i] != -1)                                       // real item or dummy?
           {
           data_P->Canvas->TextOutA(42.0*MM, (PHA+40.0+5*i)*MM, factor[crnt_fact[i]].dsn);
           sprintf(wrk_bfa, "%.1f %s", factor[crnt_fact[i]].vlu, factor[crnt_fact[i]].unt);
           data_P->Canvas->TextOutA(119.0*MM, (PHA+40.0+5*i)*MM, wrk_bfa);
           }
         }
// - - - display variables - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    for(i = 0; i < NUM_ITM; i++)                                           // display items
      {
      if(crnt_dspy[i] != -1)                                         // real item or dummy?
         {
         sprintf(wrk_bfa, "%d:", i+1);
         data_P->Canvas->TextOutA(161.0*MM, (PHA+40.0+5*i)*MM, wrk_bfa);
         data_P->Canvas->TextOutA(165.0*MM, (PHA+40.0+5*i)*MM, display[crnt_dspy[i]].dsn);
         sprintf(wrk_bfa, "%.1f %s",
                           metric_conv(crnt_dspy[i], cursor), display[crnt_dspy[i]].unt);
         data_P->Canvas->TextOutA(248.0*MM, (PHA+40.0+5*i)*MM, wrk_bfa);
         }
       }
// - - - minor headings - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    data_P->Canvas->TextOutA(42.0*MM, (PHA+35)*MM,
             "Values valid at print time, they may not represent conditions during plot.");
    sprintf(wrk_bfa, "At cusor position %02d:%02d", cursor/60, cursor%60);
    data_P->Canvas->TextOutA(161.0*MM, (PHA+35)*MM,  wrk_bfa);
// - - - headings - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
    data_P->Canvas->Font->Size = 12;
    data_P->Canvas->TextOutA(5.0*MM, (PHA+8.4)*MM, "Clinical notes:");
    data_P->Canvas->TextOutA(5.0*MM, (PHA+30)*MM, "Passport factors");
    data_P->Canvas->TextOutA(42.0*MM, (PHA+30)*MM, "Subject factors");
    data_P->Canvas->TextOutA(161.0*MM, (PHA+30)*MM, "Display variables");

    data_P->EndDoc();
    }
}
// --- respiration_I click ----------------------------------------------------------------
// displays respiration form
void __fastcall Tmain_F::respiration_IClick(TObject *Sender)
    {
    respiration_F->ShowModal();
    }
// --- file save_DB click -----------------------------------------------------------------
void __fastcall Tmain_F::save_IClick(TObject *Sender)
{
int i;
char *fe = "file exists, overwrite?";
string save_FN;                                                  // save file specification
ofstream save_FS;                                                       // save file stream
AnsiString save_F;                                                        // save file name
save_F = "subject.sbj";
main_F->save_DB->FileName = save_F;
main_F->save_DB->Filter = "Report Files (*.sbj)|*.SBJ|Text files (*.txt)|*.TXT";
save_DB->Title = "Enter subject file name";
if(main_F->save_DB->Execute())
    {
    save_F = main_F->save_DB->FileName;
    save_FN = save_F.c_str();
    if(access(save_F.c_str(), 00) == 0)                                 // does file exist?
        if(MessageDlg(fe, mtWarning, TMsgDlgButtons() << mbOK << mbCancel, 0) == mrCancel)
            return;
    sprintf(wrk_bfa, "// #001 %s   %s\n\n", save_F.c_str(), DateTimeToStr(Now()));
    strcat(wrk_bfa, "// height weight age  sex\n");                     // passport factors
    sprintf(wrk_bfb, "% 9.1f %5.1f %5.1f  %d\n\n", height.vlu, weight.vlu, age.vlu, sex);
    strcat(wrk_bfa, wrk_bfb);
    strcat(wrk_bfa, "// subject factors\n"
       "//   0       1       2       3       4       5       6       7       8       9\n");
    for(i = 0; i < num_fct; i++)                              // list subject factor values
        {
        sprintf(wrk_bfb, "%7.2f ", factor[i].vlu);
        if(!((i+1)%10))
            strcat(wrk_bfb, "\n");
        strcat(wrk_bfa, wrk_bfb);
        }
    strcat(wrk_bfa, "\n\n// display variables\n"
       "//   0       1       2       3       4       5       6       7       8       9\n");
    for(i = 0; i < num_dsy-1; i++)            // list display variable values (except last)
        {
        sprintf(wrk_bfb, "%7.2f ", display[i].vlu);
        if(!((i+1)%10))
            strcat(wrk_bfb, "\n");
        strcat(wrk_bfa, wrk_bfb);
        }
    strcat(wrk_bfa, "\n\n// other variables\n"
       "//   0       1       2       3       4       5       6       7       8       9\n");
    for(i = 0; i < num_oth; i++)                              // list other variable values
        {
        sprintf(wrk_bfb, "%7.2f ", other[i]);
        if(!((i+1)%10))
            strcat(wrk_bfb, "\n");
        strcat(wrk_bfa, wrk_bfb);
        }
    strcat(wrk_bfa, "\n\n// selected subject factors\n");
    for(i = 0; i < NUM_ITM; i++)                           // user selected subject factors
        {
        sprintf(wrk_bfb, " %d", crnt_fact[i]);
        strcat(wrk_bfa, wrk_bfb);
        }
    strcat(wrk_bfa, "\n\n// selected display variables\n");
    for(i = 0; i < NUM_ITM; i++)                         // user selected display variables
        {
        sprintf(wrk_bfb, " %d", crnt_dspy[i]);
        strcat(wrk_bfa, wrk_bfb);
        }
    strcat(wrk_bfa, "\n\n// flags and other stuff yet to be done");
    save_FS.open(save_FN.c_str());
    if(!save_FS)                                              // error: unable to open file
        MessageDlg("Sorry, cannot open subject file.", mtError, TMsgDlgButtons()<<mbOK, 0);
    else
       {
        save_FS << wrk_bfa;
        save_FS.close();
       }
   }
return;
}
// --- file load_DB click -----------------------------------------------------------------
void __fastcall Tmain_F::load_IClick(TObject *Sender)
{
int index, i;
float tf;
string load_FN;                                                  // load file specification
ifstream load_FS;                                                       // load file stream
AnsiString load_F;                                                        // load file name
load_DB->Title = "Select new subject to load";
main_F->load_DB->Filter = "Report Files (*.sbj)|*.SBJ|Text files (*.txt)|*.TXT";
if(load_DB->Execute())
    {
    load_F = main_F->load_DB->FileName;
    load_FN = load_F.c_str();
    load_FS.open(load_FN.c_str(),ios_base::in);
    if(!load_FS)                                              // error: unable to open file
        MessageDlg("Sorry, cannot open subject file.", mtError, TMsgDlgButtons()<<mbOK, 0);
    else
        {
        load_FS.getline(wrk_bfb, 100, '\n');
        if(wrk_bfb[6] != '1')
            {
            MessageDlg("Invalid subject file format.", mtError, TMsgDlgButtons()<<mbOK, 0);
            return;
            }
        wrk_bfa[0] = '\0';
         for(;;)                                              // create string of data only
             {
             if(!load_FS.getline(wrk_bfb, 100, '\n'))
                 break;
             if(!(wrk_bfb[0] == '/' && wrk_bfb[1] == '/'))
                  {
                  strcat(wrk_bfa, " ");            // ensure that there is a data separator
                  strcat(wrk_bfa, wrk_bfb);
                  }
             }
            index = satof(wrk_bfa, &s_height, 100);                        // passport data
            index += satof(wrk_bfa+index, &s_weight, 100);
            index += satof(wrk_bfa+index, &s_age, 100);
            index += satof(wrk_bfa+index, &tf, 100);
            s_sex = (int)tf;
            for(i = 0; i < num_fct; i++)                             // subject factor data
                index += satof(wrk_bfa+index, &s_factor[i], 100);
            for(i = 0; i < num_dsy-1; i++)                         // display variable data
                index += satof(wrk_bfa+index, &s_display[i].fresh, 100);
            for(i = 0; i < num_oth; i++)                             // other variable data
                index += satof(wrk_bfa+index, &s_other[i], 100);
            for(i = 0; i < NUM_ITM; i++)                            // selected factor list
                {
                index += satof(wrk_bfa+index, &tf, 100);
                crnt_fact[i] = (int)tf;
                }
            for(i = 0; i < NUM_ITM; i++)                           // selected display list
                {
                index += satof(wrk_bfa+index, &tf, 100);
                crnt_dspy[i] = (int)tf;
                }
        }
    load_FS.close();
    }
sprintf(wrk_bfa, "Ms Resi    %s", load_FN.c_str());
main_F->Caption = wrk_bfa;
sprintf(crnt_sbj, "%s", load_FN.c_str());
intro_f = true;
new_subject();
main_F->ms_resi_I->Visible = false;        // it seems we need to exercise image visibility
update();                                                                    // and updates
update();
return;
}
// --- main_F form resize -----------------------------------------------------------------
void __fastcall Tmain_F::FormResize(TObject *Sender)
{
main_F->message_P->Height  = ClientHeight-662; // makeup/lose height in client message memo
main_F->clinical_M->Height = ClientHeight-673;
main_F->control_P->Top     = ClientHeight-275;
main_F->factor_P->Top      = ClientHeight-275;
main_F->display_P->Top     = ClientHeight-275;
main_F->passport_P->Top    = ClientHeight-178;
main_F->Top = 0;                                				// position
main_F->Left = 0;
}
// --- run() ------------------------------------------------------------------------------
// run() interacts with the core model software to do 60 seconds of simulation
int run(void)
    {
    int i, j, k;
    bool alert, dead;
    constant(NREPT);                                // set constant parameters for this run
    k = NREPT - second%60;                                               // align to minute
    cursor_f = false;                                                       // reset cursor
    alert = false;                                                           // reset alert
    for(j = 0; j < k; j++)
        {
        second++;							     // next second
        if(*bag_index == RBB || *bag_index == CDA)  	      // bag rebreathing in action?
            bager(RUN, NULL, NULL); 		      // yes, make inspired gases as in bag
        if(item(model()))  	  			        // one short sample of life
            break;						 // break on internal error
        alert |= sympt();                            		   // note any new symptoms
        append();                                     // append symptom message for display
        dead = death();                                 	   // note if death occured
        append();                                       // append death message for display
        for(i = 0; i < num_dsy; i++)         // load variables into buffers (last is dummy)
            history[i][wrap(second)] = display[i].vlu;   // display is in traditional units
        for(i; i < num_dsy+num_oth; i++)                               // do others as well
            history[i][wrap(second)] = other[i-num_dsy];
        if(dead)                                              // if the last sample of life
            {
            main_F->run_B->Enabled = false;
            for(i = 0; i < num_dsy; i++)				// load dead values
                if(s_display[i].hold == false)                             // if applicable
        	    history[i][wrap(second)] = s_display[i].dead;
            j++;                                            // subtlety in cursor alignment
            break;
            }
        }
    if(alert || dead)
        Beep();                                                              // inform user
        //ADDC3=0;                              // reset bicarbonate addition index to zero
    if(second - start > int_mnt)			      // too far right for display?
        start = (second+k-j) - int_mnt; // allow for possible non-completed minute on death
    if(second > second_max)
        second_max = second;     			   // record maximum time excursion
    intro_f = false;
    update();
    main_F->back_B->Enabled = true;      // control buttons and print menu are now sensible
    main_F->forward_B->Enabled = true;
    main_F->print_I->Enabled = true;
    main_F->rerun_B->Enabled = true;
    main_F->reset_B->Enabled = true;
    return(0);
    }
// --- update() ---------------------------------------------------------------------------
// update() (re)draws work area
bool update(void)
{
int i, j;
TRect sheet(CL, 0, CR, CH);                                              // for charting on
main_F->Canvas->Brush->Color = clWhite;                                  // (re)paint sheet
main_F->Canvas->FillRect(sheet);

main_F->Canvas->Pen->Color = clBlack;
main_F->Canvas->MoveTo(CL, CH-2);                            // boundary lines, left bottom
main_F->Canvas->LineTo(CR-1, CH-2);                                         // right bottom
main_F->Canvas->LineTo(CR-1, 0);                                               // right top
main_F->Canvas->LineTo(CL, 0);                                                  // left top
main_F->Canvas->LineTo(CL, CH-2);                                            // left bottom

for(i = 0; i < 11; i++)                         // horizontial grid, 10 divisions, 11 lines
    {
    main_F->Canvas->MoveTo(GL, CH-BM-(i*GH)/10);
    main_F->Canvas->LineTo(GR, CH-BM-(i*GH)/10);
    }

main_F->Canvas->Font->Color  = clBlack;

for(i = 0; i < GR-GL; i+=15)                           // vertical grid (15 second spacing)
    {
    if(!(i%60))                                                           // integer minute
        {
        main_F->Canvas->MoveTo(i+SM, CH-BM);
        main_F->Canvas->LineTo(i+SM, TM);
        sprintf(wrk_bfa, "%d", (start+i)/60);
        main_F->Canvas->TextOutA(i+SM-4, CH-BM+2, wrk_bfa);      		    // text
        int_mnt = i;			       			// width in integer minutes
        }
    if(!((i+30)%4))                                                    // half-minute ticks
        {
        main_F->Canvas->MoveTo(i+SM, CH-BM);
        main_F->Canvas->LineTo(i+SM, CH-BM-9);
        }
    if(!((i+15)%2))                                                 // quarter-minute ticks
        {
        main_F->Canvas->MoveTo(i+SM, CH-BM);
        main_F->Canvas->LineTo(i+SM, CH-BM-6);
        }
    }
main_F->Canvas->MoveTo(GR, CH-BM);                  			    // right border
main_F->Canvas->LineTo(GR, TM);

main_F->Canvas->TextOutA(GR+8, CH-BM+2, "minutes");                    // label minute axis
calibrate();                                                    // place calibration scales
if(!intro_f)
    i = plot();                                                   // plot display variables
else
    i = 0;								  // cursor at left
if(cursor_f)			     // has update been initiated by user cursor placement?
    j = cursor;                                        // display values at cursor position
else
    {
    cursor = i;						// cursor position returned by plot
    j = second;                                             // display values at end of run
    }
draw_cursor(cursor);
update_pnls(j);
intro(intro_f);                                       // display (or not) introduction card
return(true);
}
// --- introduction card ------------------------------------------------------------------
// intro displays (or not) introduction card
bool intro(bool show)
 {
TRect card(IL, IT, IL+IW, IT+IH);                        // define introduction card extent
if(!show)
    {
    main_F->ms_resi_I->Visible = false;
    return(false);
    }
main_F->ms_resi_I->Visible = true;                                                 // image
main_F->Canvas->Brush->Color = clWhite;
main_F->Canvas->FillRect(card);                                               // paint card

main_F->Canvas->Pen->Color = clBlack;
main_F->Canvas->MoveTo(IL, IT);                              // boundary lines, left bottom
main_F->Canvas->LineTo(IL+IW, IT);                                          // right bottom
main_F->Canvas->LineTo(IL+IW, IT+IH);                                          // right top
main_F->Canvas->LineTo(IL, IT+IH);                                              // left top
main_F->Canvas->LineTo(IL, IT);                                              // left bottom
main_F->ms_resi_I->Top = IT+5;                                           // image placement
main_F->ms_resi_I->Left = IL+35;

main_F->Canvas->TextOut(IL+8, IT+200, IBS0);
main_F->Canvas->TextOut(IL+20,IT+215, IBS1A IBS1B);
main_F->Canvas->TextOut(IL+8, IT+235, IBS2);
main_F->Canvas->TextOut(IL+20,IT+250, IBS3);
main_F->Canvas->TextOut(IL+8, IT+270, IBS4);
main_F->Canvas->TextOut(IL+20,IT+285, IBS5);
sprintf(wrk_bfa, "Build: " __DATE__ "    Interface: " VERSION "    Model: %s", mdl_ver);
main_F->Canvas->TextOut(IL+170,IT+300, wrk_bfa);

return(true);
 }
// --- update_pnls() ----------------------------------------------------------------------
// update_pnls(void) updates factor and display panels values and units at nominated time
bool update_pnls(int s)
{
int i;
for(i = 0; i < NUM_ITM; i++)    		// for each subject factor and display item
    {
    if(crnt_fact[i] != -1)
        {
        if(s == second)        		  // factor data ensured valid only at current time
            fact[i].vlu->Font->Color = clBlack;
        else
            fact[i].vlu->Font->Color = clLtGray;
        sprintf(wrk_bfa, "%.2f",factor[crnt_fact[i]].vlu);
        fact[i].vlu->Text = wrk_bfa;                           			   // value
        fact[i].unit->Caption = factor[crnt_fact[i]].unt;  		           // units
        fact[i].dsn->ItemIndex = crnt_fact[i];                                // item index
        }
    else
        {
        fact[i].dsn->ItemIndex = -1;                                          // item index
        fact[i].vlu->Text = " - - - - -";                                          // value
        fact[i].unit->Caption = "";                                                // units
        }
    if(crnt_dspy[i] != -1)
        {
        sprintf(wrk_bfa, "%.2f", metric_conv(crnt_dspy[i], wrap(s)));
        dspy[i].vlu->Text = wrk_bfa;                                               // value
        dspy[i].unit->Caption = display[crnt_dspy[i]].unt;                         // units
        dspy[i].dsn->ItemIndex = crnt_dspy[i];                                // item index
        }
    }
sprintf(wrk_bfa, "%02d:%02d", s/60, s%60);
main_F->time_E->Text = wrk_bfa;
return(true);
}
// --- calibrate() ------------------------------------------------------------------------
// calibrate() provides calibration scales on either side of plot area.
bool calibrate(void)
{
float lower, upper;
char f_s[10];
int i, j, h_os;							       // horizontal offset
h_os = CL+10;  							       // left items offset
for(i = 0; i < NUM_ITM; i++)                                  // display items calibrations
    {
    if(i == 4)
        h_os = GR+10-4*CLB_SPC;          // right items offset (allow for inherited offset)
    for(j = 0; j < 11; j++)
         {
         lower = display[dspy[i].dsn->ItemIndex].lwr;
         upper = display[dspy[i].dsn->ItemIndex].upr;
         if((upper-lower) != 0)
             {
             main_F->Canvas->Pen->Color = d_clr[i];                          // line colour
             main_F->Canvas->MoveTo(h_os+i*CLB_SPC, CH-BM);
             main_F->Canvas->LineTo(h_os+i*CLB_SPC, TM-6);
             sprintf(wrk_bfa, format(upper, f_s), lower + j*0.1*(upper-lower));
             main_F->Canvas->Font->Color = d_clr[i];                         // text colour
             main_F->Canvas->TextOutA(h_os+4+i*CLB_SPC, CH-BM-9-(j*GH)/10, wrk_bfa);
             }
         }
    }
main_F->Canvas->Font->Color = clBlack;                         	     // default text colour
return(true);
}
// --- format() ---------------------------------------------------------------------------
// char *format() formats value for calibration display
char *format(float value, char *f_s)
    {
    strcpy(f_s, "%.0f");   							 // defence
    if(value >= 100)
   	strcpy(f_s, "%.0f");
    if(value < 100 && value >= 10)
	strcpy(f_s, "%.1f");
    if(value < 10 && value >= 1.0)
   	strcpy(f_s, "%0.1f");
    return(f_s);
    }
// --- plot() -----------------------------------------------------------------------------
// plot() plots current display variables. It returns time of last plot point.
int plot(void)
    {
    float scale, offset;
    int i, j, k, m, n;
    for(i = 0; i < NUM_ITM; i++)      		     // form plots of current display items
      	{
        j = crnt_dspy[i];                                       // for notation convenience
        if(display[j].lwr == display[j].upr)
            continue;					   // no display item or data error
       	main_F->Canvas->Pen->Color = d_clr[i];                	    	    // colour coded
	scale = GH/(display[j].lwr - display[j].upr);   // screen coordinates +ve downwards
      	offset = (CH-BM) - scale*display[j].lwr;
      	for(k = 0; k < second-start; k++) 			     // for each plot point
            {
            if(k == GR-SM)  	  // restrict plots to plot area (after use of back button)
            	break;
            m = wrap(start+k);
            n = wrap(start+k+1);
            if(metric_conv(j, n) < display[j].upr && metric_conv(j, n) > display[j].lwr)
                {
                main_F->Canvas->MoveTo(GL+k, metric_conv(j, m)*scale+offset);
                main_F->Canvas->LineTo(GL+k+1, metric_conv(j, n)*scale+offset);
                }
            }
        }
    return(start+k);                         		      // return time at end of plot
    }
// --- wrap() -----------------------------------------------------------------------------
// wrap() takes index to history buffer and if necessary adjusts for wrapping.
int wrap(int index)
    {
    while(index < 0)
        index += HST_LNG;
    while(index >= HST_LNG)                            /**/ // check, was (index > HST_LNG)
    	index -= HST_LNG;
    return(index);
    }
// --- draw_cursor() ----------------------------------------------------------------------
// draw_cursor(int cursor) if in valid area draws cursor line, else returns false
bool draw_cursor(int cursor)
     {
     int t;
     t = cursor - start + SM;
     if(t < SM || t > GR)                                   // is cursor outside plot area?
         {
         main_F->time_E->Color =  clYellow;                              // yes, alert user
         return(false);                                             // cursor not displayed
         }                                                  
     main_F->time_E->Color =  clWindow;                                 // remove any alert
     main_F->Canvas->Pen->Color = clRed;
     main_F->Canvas->Pen->Style = psDash;
     main_F->Canvas->MoveTo(t, CH-BM);           			   // main vertical
     main_F->Canvas->LineTo(t, TM);
     main_F->Canvas->Pen->Style = psSolid;
     main_F->Canvas->MoveTo(t-4, TM-9);             	        	       	// top tick
     main_F->Canvas->LineTo(t, TM-3);
     main_F->Canvas->LineTo(t+5, TM-10);
     main_F->Canvas->MoveTo(t-4, CH-BM+20);             		     // bottom tick
     main_F->Canvas->LineTo(t, CH-BM+14);
     main_F->Canvas->LineTo(t+5, CH-BM+21);
     main_F->Canvas->Pen->Color = clBlack;
     return(true);
     }
// --- new_subject() ----------------------------------------------------------------------
// new_subject does loads and resets for fresh run
bool new_subject(void)
    {
    int i;
    start = 0;
    second = 0;
    second_max = 0;
    cursor = 0;
    intro_f = true;
    cursor_f = false;
    main_F->clinical_M->Text = "";                                // clear clinical display
    load_new();
    main_F->sex_CB->ItemIndex = sex;
    sprintf(wrk_bfa, "%.1f", height.vlu);
    main_F->height_E->Text = wrk_bfa;
    sprintf(wrk_bfa, "%.1f", weight.vlu);
    main_F->weight_E->Text = wrk_bfa;
    sprintf(wrk_bfa, "%.1f", 10000.0*weight.vlu/(height.vlu*height.vlu));   // evaluate bmi
    main_F->bmi_E->Text = wrk_bfa;
    sprintf(wrk_bfa, "%.1f", age.vlu);
    main_F->age_E->Text = wrk_bfa;
    for(i = 0; i < num_dsy-1; i++)                  // prime history with display variables
	history[i][wrap(0)] = display[i].vlu;
    for(i; i < num_oth; i++)                                           // do others as well
        history[i][wrap(second)] = other[i];
    main_F->back_B->Enabled = false;               // suppress these items until applicable
    main_F->forward_B->Enabled = false;
    main_F->print_I->Enabled = false;
    main_F->rerun_B->Enabled = false;
    main_F->reset_B->Enabled = false;
    main_F->run_B->Enabled = true;                                            // permit run
    return(true);
    }
// --- limit() ----------------------------------------------------------------------------
// limit() adjusts user entry to lie within appropriate limits.
// It returns true if initial value was outside limits, false otherwise.
bool limit(variable *pv, TEdit *pv_E, char *format)
    {
    float vi, v;
    vi = atof(pv_E->Text.c_str());                        		  // get user entry
    v = (vi > pv->upr)? pv->upr: vi;
    v = (v < pv->lwr)? pv->lwr: v;
    pv->vlu = v;      	                 	  		           // load variable
    sprintf(wrk_bfa, format, v);      				   // load text in edit box
    pv_E->Text = wrk_bfa;
    if(v-vi)					  // if there has been a change, alert user
        alert(2, 1);
    return bool(v-vi);
    }
// --- trad_metric() ----------------------------------------------------------------------
// presents variables in mm Hg (kpa = false) or kPa (kpa = true) by swopping with alternate
void trad_metric(bool kpa)
    {
    int i, j;
    float tf, conv;
    conv = kpa? MK_CNV: 1.0/MK_CNV;
    for(i = 0; i < num_fct; i++)                                // for each factor variable
        for(j = 0; a_f_lst[j] != -1; j++)                             // for each list item
            if(i == a_f_lst[j])                              // if member of alternate list
                {                                                                  // then,
                strcpy(wrk_bfa, alt_fact[j].unt);                            // swop units,
                strcpy(alt_fact[j].unt, factor[i].unt);
                strcpy(factor[i].unt, wrk_bfa);
                tf = alt_fact[j].lwr;                                 // swop lower limits,
                alt_fact[j].lwr = factor[i].lwr;
                factor[i].lwr = tf;
                tf = alt_fact[j].upr;                                 // swop upper limits,
                alt_fact[j].upr = factor[i].upr;
                factor[i].upr = tf;
                factor[i].vlu *= conv;                                 // and convert value
                }
    for(i = 0; i < num_dsy; i++)                               // for each display variable
        for(j = 0; a_d_lst[j] != -1; j++)                             // for each list item
            if(i == a_d_lst[j])                              // if member of alternate list
                {                                                                  // then,
                strcpy(wrk_bfa, alt_dspy[j].unt);                            // swop units,
                strcpy(alt_dspy[j].unt, display[i].unt);
                strcpy(display[i].unt, wrk_bfa);
                tf = alt_dspy[j].lwr;                                 // swop lower limits,
                alt_dspy[j].lwr = display[i].lwr;
                display[i].lwr = tf;
                tf = alt_dspy[j].upr;                                 // swop upper limits,
                alt_dspy[j].upr = display[i].upr;
                display[i].upr = tf;
                }
     return;
     }
// --- metric_conv() ----------------------------------------------------------------------
// for display or plotting history display variables
float metric_conv(int v_index, int t_index)
    {
    int j;
    if(!metric)                                                          // straightforward
        return(history [v_index][t_index]);                  // if not metric world, return
    for(j = 0; a_d_lst[j] != -1; j++)                       // for each alternate list item
            if(v_index == a_d_lst[j])          // if variable is a member of alternate list
                return(MK_CNV*history [v_index][t_index]); // then do conversion and return
        return(history [v_index][t_index]);                  // if not on list return as is
    }
// --- satof() ----------------------------------------------------------------------------
// satof() loads float value and returns pointer increment
int satof(char *p, float *v, int max)
    {
    int i;
    for(i = 0; isspace(p[i]); i++)                                    // find start of data
        if(i == max)
            return(-1);
    *v = (float)atof(p);                                                         // convert
    for(i; !isspace(p[i]); i++)                                         // find end of data
        if(i == max)
            return(-1);
    return(i);
    }
// --- load_hints -------------------------------------------------------------------------
// load_hints() sets hint messages for panels and buttons.
void load_hints(void)
    {
    main_F->run_B->Hint = "run model to next minute mark";
    main_F->time_E->Hint = "cursor position mm:ss";
    main_F->hint_M->Hint = "operational alerts or hints";
    main_F->back_B->Hint = "step back in subject history";
    main_F->trad_RB->Hint = "use traditional pressure units";
    main_F->rerun_B->Hint = "strip time back to cursor";
    main_F->reset_B->Hint = "start again with same subject";
    main_F->factor_P->Hint = "select subject factor to adjust";
    main_F->metric_RB->Hint = "use SI pressure units";
    main_F->forward_B->Hint = "step forward in subject history";
    main_F->display_P->Hint = "select variables to be displayed and plotted";
    main_F->passport_P->Hint = "select passport factor to adjust";
    main_F->bmi_E->Hint = "read only";
    main_F->clinical_M->Hint = "subject symptoms, or subject comments";
    return;
    }
// --- alert() ----------------------------------------------------------------------------
void alert(int message, int bell)
    {
    char *m[] = {
    	   {""}, 							   // clear entry 0
           {" duplicate subject factor, not accepted"},              	    	       // 1
           {" entered value was outside limits, it has been adjusted"},                // 2
           {" click within time spanned by the plot"},                                 // 3
           {" cannot display negative time values"},			       	       // 4
           {" cannot back further as history capacity is limited"},		       // 5
           {" cannot display events that have not yet occurred"} 		       // 6
                };
    if(bell != 0)
        Beep();
   main_F->hint_M->Text = m[message];
   main_F->time_out_T->Interval = 5000;				      // units, millisecond
   main_F->time_out_T->Enabled = true;
   }
// --- time_out_T event -------------------------------------------------------------------
void __fastcall Tmain_F::time_out_TTimer(TObject *Sender)
    {
    main_F->time_out_T->Enabled = false;
    main_F->hint_M->Text = "";
    }
// --- append() --------------------------------------------------------------------------
// append() adds messages to clinical memo.
void append(void)
    {
    if(app_buf[0])   					      // only if message, do action
    	{
        sprintf(wrk_bfa, "%02d:%02d %s\r\n", second/60, second%60, app_buf);
        main_F->clinical_M->Text = main_F->clinical_M->Text.Insert(wrk_bfa, 1);
        app_buf[0] = '\0';
    	}
    return;
    }
// --- mark() ----------------------------------------------------------------------------
// mark() checks for overlap of chart markers on print output, returns offset if required.
float mark(int itm, float y)
    {
    int i;
    float os = 0.0;
    static float list_f[NUM_ITM];
    static bool list_b[NUM_ITM];
    if(itm < 0)                                                           // initialization
        {
        for(i = 0; i < NUM_ITM; i++)
            list_b[i] = false;
        return(0.0);
        }
    for(i = 0; i < NUM_ITM; i++)
        if((y < list_f[i]+4.0) && (y > list_f[i]-4.0) && list_b[i])   // if marker overlaps
            os += 4.2;
    list_b[itm] = true;                                                 // record this item
    list_f[itm] = y;
    return(os);
    }
// --- item() ----------------------------------------------------------------------------
// item() alerts user to non-physical values
bool item(int item)
	{
        if(!item)
            return(false);
	sprintf(wrk_bfa, "Item %d is negative - please notify authority.", item);
	MessageDlg(wrk_bfa, mtWarning, TMsgDlgButtons() << mbOK, 0);
        return(true);
        }
// --- xxxxxxxxxxxx -----------------------------------------------------------------------

